package ru.yandex.direct.db.config;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.MoreObjects;

import ru.yandex.direct.liveresource.LiveResourceEvent;
import ru.yandex.direct.liveresource.LiveResourceListener;
import ru.yandex.direct.utils.io.FileUtils;

@ParametersAreNonnullByDefault
public class DbConfigFactory implements LiveResourceListener {
    private static final String CHILDS_KEY = "CHILDS";
    private final List<DbConfigListener> listeners;
    private volatile JsonNode rootJson;
    private final Map<String, DbConfig> staticDbConfigs;

    public DbConfigFactory(String initialJson) {
        this(initialJson, null);
    }

    public DbConfigFactory(String initialJson, @Nullable Map<String, DbConfig> staticDbConfigs) {
        listeners = new CopyOnWriteArrayList<>();
        rootJson = parseJson(initialJson);
        this.staticDbConfigs = MoreObjects.firstNonNull(staticDbConfigs, Collections.emptyMap());
    }

    private static void updateDbConfig(JsonNode json, DbConfig dbConfig) {

        if (json.has("port")) {
            dbConfig.setPort(json.get("port").asInt());
        }

        if (json.has("user")) {
            dbConfig.setUser(json.get("user").asText());
        }

        if (json.has("pass")) {
            dbConfig.setPass(fetchPass(json.get("pass")));
        }

        if (json.has("engine")) {
            String engine = json.get("engine").asText();
            dbConfig.setEngine(DbConfig.Engine.valueOf(engine.toUpperCase()));
        }

        if (json.has("db")) {
            dbConfig.setDb(json.get("db").asText());
        }

        if (json.has("weight")) {
            dbConfig.setWeight(json.get("weight").asInt());
        }

        if (json.has("connect_timeout")) {
            //timeout is in seconds
            dbConfig.setConnectTimeout(json.get("connect_timeout").asDouble());
        }

        if (json.has("request_retries")) {
            dbConfig.setRequestRetries(json.get("request_retries").asInt());
        }

        if (json.has("host")) {
            JsonNode jsonHost = json.get("host");

            List<String> hosts = new ArrayList<>();
            dbConfig.setHosts(hosts);

            if (jsonHost.isArray()) {
                for (JsonNode x : jsonHost) {
                    hosts.add(x.asText());
                }
            } else if (jsonHost.isTextual()) {
                hosts.add(jsonHost.asText());
            } else {
                throw new DbConfigException("Incorrect host: " + jsonHost.toString());
            }
        }

        if (json.has("extra_users")) {
            dbConfig.setExtraUsers(fetchExtraUsers(json.get("extra_users")));
        }

        if (json.has("ssl")) {
            dbConfig.setSsl(json.get("ssl").asInt() != 0);
        }

        if (json.has("verify_ssl_certs")) {
            dbConfig.setVerify(json.get("verify_ssl_certs").asInt() != 0);
        }

        if (json.has("compression")) {
            dbConfig.setCompression(json.get("compression").asInt() != 0);
        }
    }

    private static Map<String, String> fetchExtraUsers(JsonNode extraUsers) {
        if (extraUsers instanceof ObjectNode) {
            Map<String, String> result = new HashMap<>();
            extraUsers.fieldNames().forEachRemaining(
                    user -> result.put(user, fetchPass(extraUsers.get(user).get("pass")))
            );
            return result;
        } else {
            throw new IllegalArgumentException("extra_users must be map");
        }
    }

    private static String fetchPass(JsonNode passNode) {
        if (passNode.isTextual()) {
            return passNode.asText();
        } else if (passNode.isObject() && passNode.has("file") && passNode.get("file").isTextual()) {
            return FileUtils.slurp(FileUtils.expandHome(passNode.get("file").asText())).trim();
        } else {
            throw new IllegalArgumentException("Don't know how to extract password");
        }
    }

    /**
     * @return {@code true}, если в конфигурауии есть описание для {@code dbname}, {@code false} в противном случае
     */
    public boolean has(String dbname) {
        if (staticDbConfigs.containsKey(dbname)) {
            return true;
        }

        JsonNode currentJsonNode = rootJson.get("db_config");

        for (String part : dbname.split(":")) {
            currentJsonNode = currentJsonNode.path(CHILDS_KEY).get(part);
            if (currentJsonNode == null) {
                return false;
            }
        }
        return true;
    }

    /**
     * Вернуть параметры подключения для заданной базы данных
     *
     * @param dbname Название базы данных в формате db:shardNo, например ppc:1 (Спускаться ниже сейчас бессмысленно)
     */
    public DbConfig get(String dbname) {
        if (staticDbConfigs.containsKey(dbname)) {
            return staticDbConfigs.get(dbname);
        }

        JsonNode currentJsonNode = rootJson.get("db_config");

        DbConfig dbConfig = new DbConfig();
        dbConfig.setDbName(dbname);
        updateDbConfig(currentJsonNode, dbConfig);

        for (String part : dbname.split(":")) {
            currentJsonNode = currentJsonNode.path(CHILDS_KEY).get(part);
            if (currentJsonNode == null) {
                throw new DbConfigException("No such path: " + dbname);
            }
            updateDbConfig(currentJsonNode, dbConfig);
        }

        if (currentJsonNode.has(CHILDS_KEY)) {
            currentJsonNode = currentJsonNode.path(CHILDS_KEY).get("_");
            if (currentJsonNode == null) {
                throw new DbConfigException("No leaf node for: " + dbname);
            }
            updateDbConfig(currentJsonNode, dbConfig);
        }

        if (dbConfig.getDb() == null) {
            dbConfig.setDb(dbname.split(":", 2)[0]);
        }

        return dbConfig;
    }

    /**
     * Вернуть список номеров shard-ов доступных для заданной базы данных
     *
     * @param dbname Название базы данных, например ppc
     */
    public List<String> getChildNames(String dbname) {
        if (staticDbConfigs.containsKey(dbname)) {
            throw new UnsupportedOperationException("can't get child names for dbname declared in static config");
        }

        JsonNode currentJsonNode = rootJson.get("db_config");
        for (String part : dbname.split(":")) {
            currentJsonNode = currentJsonNode.path(CHILDS_KEY).get(part);
            if (currentJsonNode == null) {
                throw new DbConfigException("No such path: " + dbname);
            }
        }

        currentJsonNode = currentJsonNode.get(CHILDS_KEY);
        if (currentJsonNode == null) {
            return Collections.emptyList();
        }

        List<String> children = new ArrayList<>(currentJsonNode.size());

        Iterator<String> it = currentJsonNode.fieldNames();
        while (it.hasNext()) {
            String child = it.next();
            if (!"_".equals(child)) {
                children.add(child);
            }
        }

        return children;
    }

    /**
     * Вернуть список номеров shard-ов доступных для заданной базы данных
     *
     * @param dbname Название базы данных, например ppc
     */
    public List<Integer> getShardNumbers(String dbname) {
        List<String> children = getChildNames(dbname);
        List<Integer> shards = new ArrayList<>(children.size());
        for (String child : children) {
            Integer number;
            try {
                number = Integer.valueOf(child, 10);
            } catch (NumberFormatException e) {
                // not a shard number
                continue;
            }
            shards.add(number);
        }
        return shards;
    }

    @Override
    public void update(LiveResourceEvent event) {
        rootJson = parseJson(event.getCurrentContent());
        notifyListeners();
    }

    public void addListener(DbConfigListener listener) {
        listeners.add(listener);
    }

    public void removeListener(DbConfigListener listener) {
        listeners.remove(listener);
    }

    public String getCurrentConfig() {
        return rootJson.toString();
    }

    private void notifyListeners() {
        for (DbConfigListener listener : listeners) {
            listener.update(new DbConfigEvent());
        }
    }

    @Nonnull
    private JsonNode parseJson(String jsonString) {
        try {
            JsonNode json = new ObjectMapper().readTree(jsonString);
            if (json == null) {
                throw new DbConfigException("Parsed null from: " + jsonString);
            }
            return json;
        } catch (IOException ex) {
            throw new DbConfigException("Can't parse db-config: " + jsonString, ex);
        }
    }

}
