package ru.yandex.direct.env;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import org.apache.commons.lang3.StringUtils;


/**
 * EnvironmentType defines possible environment types
 */
public enum EnvironmentType {

    PRODUCTION("Продакшн", false, null),
    ROPROD("Продакшен на чтение", false, PRODUCTION),
    PRESTABLE("Среда ограниченного тестирования", false, PRODUCTION),
    TESTING("Тестовая среда на БД ts-1", false, PRODUCTION),
    TESTING2("Тестовая среда на БД ts-2", false, TESTING),
    DEVTEST("Бета на базе devtest", false, TESTING),
    DEV7("Бета на базе dev7", false, DEVTEST),
    LOADTEST("Бета с loadtest", false, DEVTEST),
    DEVELOPMENT("Разработка на локальной машине", false, DEVTEST),
    DB_TESTING("Юнит-тестирование на базе данных", false, DEVELOPMENT),
    SANDBOX("Песочница", true, PRODUCTION),
    SANDBOX_TESTING("Тестовая среда песочницы", true, SANDBOX),
    SANDBOX_DEVTEST("Песочница на devtest бете", true, SANDBOX_TESTING),
    SANDBOX_DEV7("Песочница на dev7 бете", true, SANDBOX_TESTING),
    SANDBOX_DEVELOPMENT("Песочница на локальной машине", true, SANDBOX_TESTING);

    static final EnvironmentType DEFAULT = DEVELOPMENT;

    private static final Map<EnvironmentType, EnvironmentType> SANDBOX_MAP = Map.of(
            PRODUCTION, SANDBOX,
            TESTING, SANDBOX_TESTING,
            DEVELOPMENT, SANDBOX_DEVELOPMENT
    );

    /**
     * Маппинг между конфигурациями и их историческими названиями
     */
    private static final BiMap<EnvironmentType, String> LEGACY_NAMES_MAP = ImmutableBiMap.of(
            TESTING, "test",
            TESTING2, "test2",
            SANDBOX_TESTING, "sandboxtest",
            SANDBOX_DEVTEST, "sandboxdevtest",
            SANDBOX_DEV7, "sandboxdev7"
    );

    /**
     * Маппинг между конфигурациями из environment.type и доступными для них уточнениями из environment.name
     */
    private static final Map<EnvironmentType, Set<EnvironmentType>> CLARIFICATION_MAP = Map.of(
            DEVELOPMENT, Set.of(DEVTEST, DEV7),
            TESTING, Set.of(TESTING, TESTING2, LOADTEST)
    );

    private final String description;
    private final boolean isSandbox;
    private final EnvironmentType fallback;

    EnvironmentType(String description, boolean isSandbox, EnvironmentType fallback) {
        this.description = description;
        this.isSandbox = isSandbox;
        this.fallback = fallback;
    }

    /**
     * Check, is provided environment type is sandbox
     *
     * @return true, if environment is one of sandbox: production, testing or development
     */
    public boolean isSandbox() {
        return isSandbox;
    }

    public boolean isProductionSandbox() {
        return this == SANDBOX;
    }

    public boolean isProduction() {
        return this == PRODUCTION;
    }

    public boolean isProductionOrPrestable() {
        return this == PRODUCTION || this == PRESTABLE;
    }

    public boolean isTesting() {
        return this == TESTING || this == TESTING2 || this == SANDBOX_TESTING;
    }

    public boolean isDevelopment() {
        return this == DEVELOPMENT || this == SANDBOX_DEVELOPMENT || this == DB_TESTING;
    }

    public boolean isBeta() {
        return this == DEV7 || this == DEVTEST || this == DEVELOPMENT
                || this == SANDBOX_DEV7 || this == SANDBOX_DEVTEST || this == SANDBOX_DEVELOPMENT
                || this == DB_TESTING || this == LOADTEST;
    }

    /**
     * Check if provided environment is a subset of another environment
     *
     * @param env - environment type
     * @return true if environment is a subset of another environment
     */
    public boolean isClarificationOf(EnvironmentType env) {
        return CLARIFICATION_MAP.getOrDefault(env, Set.of()).contains(this);
    }

    /**
     * Strip and uppercase string, return corresponding constant
     *
     * @param env - string representation of environment type
     * @return - null if env is null or whitespace-only, corresponding environment type otherwise
     * @throws IllegalArgumentException if no such constant
     */
    @Nullable
    static EnvironmentType resolve(@Nullable String env) {
        if (env == null) {
            return null;
        }
        String strippedStr = StringUtils.strip(env);
        if (strippedStr.isEmpty()) {
            return null;
        }
        if (LEGACY_NAMES_MAP.containsValue(strippedStr)) {
            return LEGACY_NAMES_MAP.inverse().get(strippedStr);
        }
        return valueOf(strippedStr.toUpperCase());
    }

    /**
     * @return - список, состоящий из оригинального себя и иерархии фолбеков
     * (в какое окружение смотреть, если нет конфига для текущего)
     */
    @Nonnull
    public List<EnvironmentType> getHierarchy() {
        List<EnvironmentType> ret = new ArrayList<>();

        EnvironmentType e = this;
        while (e != null) {
            ret.add(e);
            e = e.fallback;
        }

        return ret;
    }


    /**
     * Имеет ли текущее окружение SANDBOX-версию?
     */
    public boolean hasSandbox() {
        return SANDBOX_MAP.containsKey(this);
    }

    /**
     * Получить окружение, соответствующее песочнице для текущего
     */
    @Nullable
    public EnvironmentType getSandbox() {
        return SANDBOX_MAP.get(this);
    }

    /**
     * Описание окружения
     */
    public String getDescription() {
        return description;
    }

    /**
     * Получить историческое название окружения
     */
    public String getLegacyName() {
        return LEGACY_NAMES_MAP.getOrDefault(this, this.name().toLowerCase());
    }

}
