package ru.yandex.direct.env;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.io.FileUtils;

/**
 * Get environment type from one of several source:
 * - system property
 * - system environment
 * - filename (usually delivered by one of yandex-environment debian package)
 */
class EnvironmentTypeReader {
    private static final Logger logger = LoggerFactory.getLogger(EnvironmentTypeReader.class);

    static final String PROPERTY_NAME = "yandex.environment.type";
    static final String ENV_NAME = "YANDEX_ENVIRONMENT";
    static final String FILE_NAME = "/etc/yandex/environment.type";
    static final String CLARIFICATION_FILE_NAME = "/etc/yandex/environment.name";

    // если есть это файл - значит мы на сервере с песочницей Директа
    private static final String DIRECT_SANDBOX_FLAG_FILE = "/etc/yandex/environment.direct_sandbox";

    static class ReadException extends RuntimeException {
        ReadException(String msg, Throwable e) {
            super(msg, e);
        }
    }

    private EnvironmentTypeReader() {
        // no initialization
    }

    /**
     * Get environment type with default property name, default environment variable name,
     * default file name
     *
     * @throws ReadException if io error occurred or unsupported enviroment type defined
     */
    @Nonnull
    static EnvironmentType read() {
        return read(PROPERTY_NAME, ENV_NAME, FILE_NAME, CLARIFICATION_FILE_NAME);
    }

    /**
     * Get environment type with:
     *
     * @param propName    - name of system property
     * @param envName     - name of system environment variable
     * @param fileName    - name of file with environment data
     * @param addFileName - name of file with additional environment data
     * @throws ReadException if io error occurred or unsupported environment type defined
     */
    @Nonnull
    static EnvironmentType read(String propName, String envName, String fileName, String addFileName) {
        try {
            var env = EnvironmentType.resolve(readFromProperties(propName));
            if (env == null) {
                env = EnvironmentType.resolve(readFromEnv(envName));
            }
            if (env == null) {
                env = readFromFile(fileName, addFileName);
                // если мы в песочнице - модифицируем env сооветствующим образом
                if (env != null && isDirectSandbox() && env.hasSandbox()) {
                    env = env.getSandbox();
                }
            }
            if (env == null) {
                env = EnvironmentType.DEFAULT;
            }
            return env;
        } catch (IOException | IllegalArgumentException e) {
            throw new ReadException("Can't get environment", e);
        }
    }

    private static String readFromProperties(String propName) {
        return System.getProperty(propName);
    }

    private static String readFromEnv(String envName) {
        return System.getenv(envName);
    }

    private static EnvironmentType readFromFile(String fileName, String addFileName) throws IOException {
        var path = Path.of(fileName);
        if (Files.notExists(path)) {
            return null;
        }
        var env = EnvironmentType.resolve(FileUtils.slurp(path));
        var addPath = Path.of(addFileName);
        if (env != null && Files.exists(addPath)) {
            try {
                var clarifiedEnv = EnvironmentType.resolve(FileUtils.slurp(addPath));
                if (clarifiedEnv != null && clarifiedEnv.isClarificationOf(env)) {
                    env = clarifiedEnv;
                } else {
                    logger.warn("Environment value \"{}\" from /etc/yandex/environment.name does not correspond to " +
                            "value \"{}\" in /etc/yandex/environment.type", clarifiedEnv, env);
                }
            } catch (IllegalArgumentException e) {
                logger.warn("Exception caught while parsing environment value from /etc/yandex/environment.name: ", e);
            }
        }
        return env;
    }

    /**
     * В Директе sandbox это не отдельное окружение, а модификатор окружения,
     * информация о том, что среда - песочница определяется по впециальному файлу
     */
    private static boolean isDirectSandbox() {
        return new File(DIRECT_SANDBOX_FLAG_FILE).exists();
    }
}
