package ru.yandex.direct.liveresource;

import java.net.URI;
import java.net.URISyntaxException;

import ru.yandex.direct.utils.io.FileUtils;

/**
 * Factory for creating {@link LiveResource} objects, type of concrete LiveResource determined by prefix of resourceUrl
 */
public class LiveResourceFactory {
    private static final LiveResource EMPTY = new EmptyResource();

    private LiveResourceFactory() {
    }

    /**
     * Create concrete {@link LiveResource} object, type determined by prefix of resourceUrl
     *
     * @param resource -  memory content, if starts with memory//:
     *                 or url of resource, two types of urls are supported:
     *                 file:///path/to/file
     *                 classpath:///path/in/classpath
     * @return @LiveResource object
     * @throws IllegalArgumentException for unsupported protocol
     */
    public static LiveResource get(String resource) {
        if (resource.startsWith("memory://")) {
            return new MemoryLiveResource(resource.replaceFirst("memory://", ""));
        }

        try {
            return get(new URI(resource));
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("Incorrect resource URL: " + resource, e);
        }
    }

    private static LiveResource get(URI uri) {
        String scheme = uri.getScheme();
        if ("classpath".equals(scheme)) {
            return new ClasspathLiveResource(uri.getPath());
        } else if ("file".equals(scheme)) {
            var path = (uri.getAuthority() != null ? uri.getAuthority() : "") + uri.getPath();
            return new FileLiveResource(FileUtils.expandHome(path));
        } else {
            throw new IllegalArgumentException("Incorrect resource URL, unsupported scheme: " + uri);
        }
    }

    /**
     * Get resource, which returns empty content
     */
    public static LiveResource emptyResource() {
        return EMPTY;
    }
}
