package ru.yandex.direct.liveresource;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledFuture;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.util.Assert;

/**
 * Class for watching for changes in LiveResource and notifying listeners
 */
public class PollingLiveResourceWatcher implements LiveResourceWatcher {
    private static final Logger logger = LoggerFactory.getLogger(PollingLiveResourceWatcher.class);

    private final LiveResource liveResource;
    private final TaskScheduler taskScheduler;
    private final long checkRate;
    private final List<LiveResourceListener> listeners = new CopyOnWriteArrayList<>();

    private String previousContent;
    private ScheduledFuture<?> scheduledFuture;

    public PollingLiveResourceWatcher(LiveResource liveResource,
                                      String initialContent,
                                      TaskScheduler taskScheduler,
                                      long checkRate) {
        Assert.notNull(liveResource, "liveResource is required");
        Assert.notNull(initialContent, "initialContent is required");
        Assert.notNull(taskScheduler, "taskScheduler is required");
        Assert.isTrue(checkRate > 0, "checkRate must be greater than 0");
        this.liveResource = liveResource;
        this.previousContent = initialContent;
        this.taskScheduler = taskScheduler;
        this.checkRate = checkRate;
    }

    public void addListener(LiveResourceListener listener) {
        listeners.add(listener);
    }

    public String watch() {
        scheduledFuture = taskScheduler.scheduleAtFixedRate(() -> {
            logger.debug("Checking for LiveResource content change for {}", liveResource.getLocation());

            String currentContent;
            try {
                currentContent = liveResource.getContent();
            } catch (Exception e) {
                logger.error("Error reading LiveResource during checking content change {}",
                        liveResource.getLocation(), e);
                return;
            }

            if (!StringUtils.equals(previousContent, currentContent)) {
                previousContent = currentContent;
                logger.debug("LiveResource content changed, notifying listeners ({})", liveResource.getLocation());
                notifyListeners(currentContent);
            }
        }, checkRate);
        return previousContent;
    }

    private void notifyListeners(String currentContent) {
        LiveResourceEvent event = new LiveResourceEvent(currentContent);
        for (LiveResourceListener listener : listeners) {
            try {
                listener.update(event);
            } catch (RuntimeException ex) {
                logger.error("listener {} failed to process LiveResourceEvent for location {}",
                        listener, liveResource.getLocation(), ex);
            }
        }
    }

    @Override
    public void close() {
        scheduledFuture.cancel(true);
    }

    @Override
    public LiveResource getLiveResource() {
        return liveResource;
    }
}
