package ru.yandex.direct.logging;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.pattern.ConverterKeys;
import org.apache.logging.log4j.core.pattern.LogEventPatternConverter;
import org.apache.logging.log4j.core.pattern.PatternConverter;
import org.apache.logging.log4j.util.ReadOnlyStringMap;

/**
 * Специализированная версия {@link org.apache.logging.log4j.core.pattern.MdcPatternConverter}.
 * <p>
 * Отличия:
 * <ul>
 * <li>Умеет выводить только 1 поле за раз</li>
 * <li>Может настраиваться значением по-умолчанию, к примеру {@code %CUSTOMMDC{field,default('defaultFieldValue')} }
 * </li>
 * </ul>
 */
@Plugin(name = "CustomMdcPatternConverter", category = PatternConverter.CATEGORY)
@ConverterKeys({CustomMdcPatternConverter.PRIMARY_KEY})
public class CustomMdcPatternConverter extends LogEventPatternConverter {
    public static final String PRIMARY_KEY = "CUSTOMMDC";
    private final CustomMdcOptions mdcOptions;

    private CustomMdcPatternConverter(final String[] options) {
        super(options != null && options.length > 0 ?
                String.format("%s{%s}", CustomMdcPatternConverter.PRIMARY_KEY, options[0]) :
                PRIMARY_KEY, "mdc");
        this.mdcOptions = CustomMdcOptions.parse(options);
    }

    @Override
    public void format(LogEvent event, StringBuilder toAppendTo) {
        ReadOnlyStringMap contextMap = event.getContextData();
        String key = mdcOptions.key();
        String keyValue = contextMap.getValue(key);
        if (keyValue == null) {
            keyValue = GlobalCustomMdc.getValue(key);
        }
        if (keyValue == null) {
            keyValue = mdcOptions.defaultValue();
        }
        if (keyValue != null) {
            toAppendTo.append(keyValue);
        }
    }

    public static CustomMdcPatternConverter newInstance(final String[] options) {
        return new CustomMdcPatternConverter(options);
    }

    static class CustomMdcOptions {
        static final Pattern DEFAULT_VALUE_PATTERN = Pattern.compile("default\\('(.*)'\\)");
        private final String key;
        private final String defaultValue;

        CustomMdcOptions(String key, String defaultValue) {
            this.key = key;
            this.defaultValue = defaultValue;
        }

        public static CustomMdcOptions parse(String[] options) {
            if (options == null || options.length == 0) {
                throw new IllegalArgumentException("options are mandatory");
            }
            String opt = options[0];
            String[] tokens = opt.split(",");
            if (tokens.length > 2) {
                throw new IllegalArgumentException("too many option tokens: " + opt);
            }
            String key = tokens[0];
            String defaultValue = tokens.length == 2 ? sanitizeDefaultValue(tokens[1]) : null;
            return new CustomMdcOptions(key, defaultValue);
        }

        private static String sanitizeDefaultValue(String src) {
            Matcher m = DEFAULT_VALUE_PATTERN.matcher(src);
            if (!m.matches()) {
                throw new IllegalArgumentException("illegal default value expression: " + src);
            }
            String extractedToken = m.group(1);
            return extractedToken;
        }

        public String key() {
            return key;
        }

        public String defaultValue() {
            return defaultValue;
        }
    }
}
