package ru.yandex.direct.logging;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.StringJoiner;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.config.Node;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.layout.AbstractStringLayout;
import org.jetbrains.annotations.NotNull;

import ru.yandex.direct.env.Environment;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceMdcAdapter;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.tracing.util.TraceCommentVarsHolder;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.SystemUtils;
import ru.yandex.direct.version.DirectVersion;

import static ru.yandex.direct.utils.CommonUtils.nvl;

@Plugin(name = "ErrorBoosterLayout", category = Node.CATEGORY, elementType = Layout.ELEMENT_TYPE, printObject = true)
public class ErrorBoosterLayout extends AbstractStringLayout {
    private final ErrorMessageNormalizer errorMessageNormalizer = new ErrorMessageNormalizer();

    protected ErrorBoosterLayout(Charset charset) {
        super(charset);
    }

    @PluginFactory
    public static ErrorBoosterLayout createLayout() {
        return new ErrorBoosterLayout(StandardCharsets.UTF_8);
    }

    @Override
    public String toSerializable(LogEvent event) {
        try (TraceProfile profile = Trace.current().profile("error_booster:serialization")) {
            return JsonUtils.toJson(new ErrorBoosterLogRecord(event)) + '\n';
        }
    }

    private class ErrorBoosterLogRecord {
        // сериализуется в json: https://wiki.yandex-team.ru/error-booster/error-how-to/#kakispolzovat
        @JsonProperty("project")
        private String project = "direct";

        @JsonProperty("message")
        private String message;

        @JsonProperty("timestamp")
        private long timestamp;

        @JsonProperty("reqid")
        private String reqid;

        @JsonProperty("language")
        private String language = "java";

        @JsonProperty("service")
        private String service;

        @JsonProperty("stack")
        private String stack;

        @JsonProperty("source")
        private String source;

        @JsonProperty("method")
        private String method;

        @JsonProperty("host")
        private String host;

        @JsonProperty("env")
        private String environment;

        @JsonProperty("fingerprint")
        private String fingerprint;

        @JsonProperty("yandexuid")
        private String yandexuid;

        @JsonProperty("version")
        private String version;

        @JsonProperty("dc")
        private String dc;

        @JsonProperty("additional")
        private Map<String, String> additional = new HashMap<>();

        public ErrorBoosterLogRecord(LogEvent event) {
            timestamp = System.currentTimeMillis();
            host = SystemUtils.hostname();

            dc = SystemUtils.hostDatacenter();
            environment = Environment.getCached().toString().toLowerCase();
            try {
                version = DirectVersion.getVersion();
            } catch (RuntimeException e) {
                System.err.println("Can't determine version fo ErrorBooster: " + e.getMessage());
            }

            message = createErrorMessage(event);
            fingerprint = createFingerprint(event, message);
            stack = formatStackTrace(event);

            var trace = Trace.currentOrNull();
            if (trace == null) {
                reqid = "0";
                service = nvl(GlobalCustomMdc.getValue(TraceMdcAdapter.SERVICE_KEY), "unknown");
                method = nvl(GlobalCustomMdc.getValue(TraceMdcAdapter.METHOD_KEY), "unknown");
            } else {
                reqid = Long.toString(trace.getSpanId());
                service = trace.getService();
                method = trace.getMethod();
            }
            source = event.getLoggerName();

            if (event.getThrown() != null) {

                if(event.getThrown().getSuppressed().length > 0) {
                    var suppressed = new ArrayList<>(Arrays.asList(event.getThrown().getSuppressed()));
                    try {
                        additional.put("suppressed", JsonUtils.toDeterministicJson(suppressed));
                    } catch (Exception e) {
                        System.err.println("Can't serialize suppressed exceptions: " + e.getMessage());
                    }
                }
                additional.put("thrownClass", event.getThrown().getClass().getName());
                var rootCause = ExceptionUtils.getRootCause(event.getThrown());
                if (rootCause != null) {
                    additional.put("rootCauseClass", rootCause.getClass().getName());
                    additional.put("rootCauseMessage", rootCause.getMessage());
                    additional.put("rootCauseNormalizedMessage",
                            errorMessageNormalizer.normalize(rootCause.getMessage()));
                }
            }
            additional.put("fingerprint", fingerprint);

            yandexuid = TraceCommentVarsHolder.get().getOperator();
        }

        private String createErrorMessage(LogEvent event) {
            var sb = new StringBuilder(event.getMessage().getFormattedMessage());
            getExceptionsChain(event).forEach(
                    e -> sb.append("; Caused by ")
                            .append(e.getMessage())
                            .append(" @ ")
                            .append(e.getStackTrace().length > 0 ? e.getStackTrace()[0].getClassName() : "unknown")
            );
            return sb.toString();
        }

        private String createFingerprint(LogEvent event, String message) {
            var joiner = new StringJoiner("|");
            joiner.add(event.getLoggerName());
            joiner.add(errorMessageNormalizer.normalize(message));
            joiner.add(createExceptionFingerprint(event));
            return joiner.toString();
        }

        @NotNull
        private String createExceptionFingerprint(LogEvent event) {
            return getExceptionsChain(event).stream()
                    .map(e -> {
                        var stackTrace = e.getStackTrace();
                        return (stackTrace.length > 0 ? stackTrace[0].getClassName() : "unknown")
                                + "-"
                                + errorMessageNormalizer.normalize(e.getMessage());
                    })
                    .collect(Collectors.joining(";"));
        }

        @NotNull
        private ArrayList<Throwable> getExceptionsChain(LogEvent event) {
            var exceptions = new ArrayList<Throwable>();
            var thrown = event.getThrown();
            int i = 0;

            while (thrown != null && i++ < 4) {
                exceptions.add(thrown);
                thrown = thrown.getCause();
            }
            return exceptions;
        }

        private String formatStackTrace(LogEvent event) {
            if (event.getThrown() != null) {
                return ExceptionUtils.getStackTrace(event.getThrown());
            }

            StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();

            var joiner = new StringJoiner("\n");
            // вырезать логирование из стектрейса
            boolean skipElement = true;
            for (StackTraceElement el : stackTrace) {
                if (!skipElement) {
                    joiner.add("\tat " + el);
                } else if (el.getClassName().equals("org.apache.logging.slf4j.Log4jLogger")) {
                    skipElement = false;
                }
            }
            return joiner.toString();
        }
    }
}
