package ru.yandex.direct.logging;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.util.PluginManager;
import org.apache.logging.log4j.core.config.plugins.util.PluginType;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Утилиты для регистрации плагинов log4j
 */
public class Log4jPluginRegistrar {
    public static final List<Class<?>> PLUGINS = Arrays.asList(
            HostnamePatternConverter.class,
            CustomMdcPatternConverter.class,
            LoggingMaskingConverter.class,
            StatsFilter.class,
            LoggerNameFilter.class,
            LayoutRewritePolicy.class,
            RewriteOverflowingAppender.class
    );

    private Log4jPluginRegistrar() {
    }

    /**
     * Добавляет пакеты требуемых плагинов в Log4j
     */
    public static void registerPluginPackages() {
        List<String> tmp = mapList(PLUGINS, cls -> cls.getPackage().getName());
        Set<String> pluginPackages = new HashSet<>(tmp);
        PluginManager.addPackages(pluginPackages);
    }

    /**
     * Проверяет, успешно ли загружаются требуемые плагины
     *
     * @throws IllegalStateException если плагины не загружаются
     */
    public static void checkPluginsRegistration() {
        Map<String, Set<Class<?>>> pluginCache = new HashMap<>();
        for (Class<?> pluginClass : PLUGINS) {
            Plugin pluginDescriptor = pluginClass.getAnnotation(Plugin.class);
            final String pluginCategory = pluginDescriptor.category();
            Set<Class<?>> categoryPlugins;
            if (pluginCache.containsKey(pluginCategory)) {
                categoryPlugins = pluginCache.get(pluginCategory);
            } else {
                PluginManager manager = new PluginManager(pluginCategory);
                manager.collectPlugins();
                Set<Class<?>> collectedClasses = manager
                        .getPlugins()
                        .values()
                        .stream()
                        .map(PluginType::getPluginClass)
                        .collect(Collectors.toSet());
                pluginCache.put(pluginCategory, collectedClasses);
                categoryPlugins = collectedClasses;
            }
            if (!categoryPlugins.contains(pluginClass)) {
                throw new IllegalArgumentException("plugin " + pluginClass + " failed to register");
            }
        }
    }
}
