package ru.yandex.direct.logging;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Comparator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.Marker;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.Logger;
import org.apache.logging.log4j.core.config.Node;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.filter.AbstractFilter;
import org.apache.logging.log4j.message.Message;

/**
 * Фильтр для отлова логеров по названию
 * работает аналогично {@link org.apache.logging.log4j.core.filter.RegexFilter}
 * нужен для того, чтобы в дев окружении задать чёрный список логеров
 */
@Plugin(name = "LoggerNameFilter", category = Node.CATEGORY, elementType = Filter.ELEMENT_TYPE, printObject = true)
public final class LoggerNameFilter extends AbstractFilter {

    private static final int DEFAULT_PATTERN_FLAGS = 0;
    private final Pattern pattern;

    private LoggerNameFilter(final Pattern pattern, final Filter.Result onMatch, final Filter.Result onMismatch) {
        super(onMatch, onMismatch);
        this.pattern = pattern;
    }

    @Override
    public Filter.Result filter(final Logger logger, final Level level, final Marker marker, final String msg,
                                final Object... params) {
        return filter(logger.getName());
    }

    @Override
    public Filter.Result filter(final Logger logger, final Level level, final Marker marker, final Object msg,
                                final Throwable t) {
        return filter(logger.getName());
    }

    @Override
    public Filter.Result filter(final Logger logger, final Level level, final Marker marker, final Message msg,
                                final Throwable t) {
        return filter(logger.getName());
    }

    @Override
    public Filter.Result filter(final LogEvent event) {
        return filter(event.getLoggerName());
    }

    private Filter.Result filter(final String name) {
        if (name == null) {
            return onMismatch;
        }
        final Matcher m = pattern.matcher(name);
        return m.matches() ? onMatch : onMismatch;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append(", pattern=").append(pattern.toString());
        return sb.toString();
    }

    /**
     * Create a Filter that matches a regular expression.
     *
     * @param regex        The regular expression to match.
     * @param patternFlags An array of Stirngs where each String is a {@link Pattern#compile(String, int)}
     *                     compilation flag.
     * @param match        The action to perform when a match occurs.
     * @param mismatch     The action to perform when a mismatch occurs.
     * @return The RegexFilter.
     * @throws IllegalAccessException
     * @throws IllegalArgumentException
     */
    @PluginFactory
    public static LoggerNameFilter createFilter(
            //@formatter:off
            @PluginAttribute("regex") final String regex,
            @PluginElement("PatternFlags") final String[] patternFlags,
            @PluginAttribute("onMatch") final Filter.Result match,
            @PluginAttribute("onMismatch") final Filter.Result mismatch)
    //@formatter:on
            throws IllegalArgumentException, IllegalAccessException {
        if (regex == null) {
            LOGGER.error("A regular expression must be provided for RegexFilter");
            return null;
        }
        return new LoggerNameFilter(Pattern.compile(regex, toPatternFlags(patternFlags)), match, mismatch);
    }

    private static int toPatternFlags(final String[] patternFlags) throws IllegalArgumentException,
            IllegalAccessException {
        if (patternFlags == null || patternFlags.length == 0) {
            return DEFAULT_PATTERN_FLAGS;
        }
        final Field[] fields = Pattern.class.getDeclaredFields();
        final Comparator<Field> comparator = Comparator.comparing(Field::getName);
        Arrays.sort(fields, comparator);
        final String[] fieldNames = new String[fields.length];
        for (int i = 0; i < fields.length; i++) {
            fieldNames[i] = fields[i].getName();
        }
        int flags = DEFAULT_PATTERN_FLAGS;
        for (final String test : patternFlags) {
            final int index = Arrays.binarySearch(fieldNames, test);
            if (index >= 0) {
                final Field field = fields[index];
                flags |= field.getInt(Pattern.class);
            }
        }
        return flags;
    }
}
