package ru.yandex.direct.logging;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.ConfigurationFactory;
import org.apache.logging.log4j.internal.LogManagerStatus;
import org.apache.logging.slf4j.Log4jLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.env.Environment;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.tracing.TraceMdcAdapter;
import ru.yandex.direct.version.DirectVersion;

/**
 * Инициализирует log4j. Имя используемого конфига строится на основе {@link Environment#getCached()}
 * (либо {@link EnvironmentType#PRODUCTION}), см {@link #configNameForEnvironment(EnvironmentType)}.
 * Инициализация пропускаеся если выставлена переменная среды
 * {@link ConfigurationFactory#CONFIGURATION_FILE_PROPERTY}
 * <p>
 * Так же проверяется бекенд, выбранный slf4j, если не log4j2 - кидаем исключение.
 */
public class LoggingInitializer {
    static {
        // Если конфигурация была загружена до этого момента логирование может работать некорректно
        if (LogManagerStatus.isInitialized() && !LoggingInitializerUtils.isReinitializationAllowed()) {
            throw new IllegalStateException("Logging was initialized before loading LoggingInitializer");
        }

        // MDC thread inheritance for log4j
        System.setProperty("isThreadContextMapInheritable", "true");
        // hibernate-validator должен использовать slf4j
        System.setProperty("org.jboss.logging.provider", "slf4j");

        Log4jPluginRegistrar.registerPluginPackages();
        Log4jPluginRegistrar.checkPluginsRegistration();

        // Здесь первый раз инициализируется контекст log4j2
        logger = LoggerFactory.getLogger(LoggingInitializer.class);
        VERSION_LOGGER = LoggerFactory.getLogger("PROGRAM_VERSION.log");
    }

    private static final Logger logger;
    private static final Logger VERSION_LOGGER;

    // работаем только с log4j
    private static final Class<Log4jLogger> REQUIRED_BACKEND = Log4jLogger.class;

    private LoggingInitializer() {
    }

    public static void initialize(LoggingInitializerParams params, Map<String, String> ctx) {
        GlobalCustomMdc.putAll(ctx);
        initialize(params);
    }

    public static void initialize(LoggingInitializerParams params, String serviceName) {
        GlobalCustomMdc.put(TraceMdcAdapter.SERVICE_KEY, serviceName);
        initialize(params);
    }

    public static void initialize(LoggingInitializerParams params, String serviceName, String methodName) {
        GlobalCustomMdc.put(TraceMdcAdapter.SERVICE_KEY, serviceName);
        GlobalCustomMdc.put(TraceMdcAdapter.METHOD_KEY, methodName);
        initialize(params);
    }

    private static void initializeFromUri(URI configUri) {
        checkLoggingBackend();
        if (System.getProperty(ConfigurationFactory.CONFIGURATION_FILE_PROPERTY) == null && configUri != null) {
            LoggerContext context = (LoggerContext) LogManager.getContext(false);
            context.setConfigLocation(configUri);
            context.updateLoggers();
            Log4jPluginRegistrar.checkPluginsRegistration();
        }
        logProgramVersion();
    }

    /**
     * Инициализириует log4j конфигом из каталога.
     */
    private static void initializeFromDirectory(File directory) {
        URI configURI = configUriForDirectory(directory, Environment.getCached());
        if (configURI == null) {
            configURI = configUriForDirectory(directory, EnvironmentType.PRODUCTION);
        }
        if (configURI == null) {
            throw new IllegalStateException("can't locate log4j config candidate");
        }
        initializeFromUri(configURI);
        logger.info("initialized from {}", configURI);
    }

    /**
     * Инициализириует log4j конфигом из classpath.
     */
    public static void initializeFromClasspath() {
        URI configURI = Environment.getCached().getHierarchy().stream()
                .map(LoggingInitializer::configUriForClassPath)
                .filter(Objects::nonNull)
                .findFirst()
                .orElse(null);

        if (configURI == null) {
            throw new IllegalStateException("can't locate log4j config candidate");
        }
        initializeFromUri(configURI);
        logger.info("initialized from {}", configURI);
    }

    private static void logProgramVersion() {
        VERSION_LOGGER.info(getEnvironmentInfo() + DirectVersion.getVcsVersion().getProgramSvnVersion());
    }

    private static String getEnvironmentInfo() {
        return "Current environment: " + Environment.getCached().name() + "\n\n";
    }

    private static void checkLoggingBackend() {
        if (logger.getClass() != REQUIRED_BACKEND) {
            throw new IllegalStateException("Incorrect logger backend: " + logger.getClass().getCanonicalName());
        }
    }

    /**
     * Инициализириует log4j конфигом. Используется каталог, указанный в командной строке,
     * либо конфиг из classpath если в командной строке не было соответствующей опции
     *
     * @param loggingParams 'метаконфиг' логирующей подсистемы
     */
    public static void initialize(LoggingInitializerParams loggingParams) {
        if (loggingParams.getConfigsDirectory() == null) {
            LoggingInitializer.initializeFromClasspath();
        } else {
            LoggingInitializer.initializeFromDirectory(new File(loggingParams.getConfigsDirectory()));
        }
        Thread.setDefaultUncaughtExceptionHandler((t, e) -> {
            LoggerFactory.getLogger(LoggingInitializer.class)
                    .error("Uncaught Exception in thread " + t.getName(), e);
        });
    }

    /**
     * Обёртка, которую имеет смысл использовать в Main-классах, чтобы System-проперти из static-секции
     * успели установиться до инициализации log4j
     */
    public static Logger getLogger(Class<?> clazz) {
        return LoggerFactory.getLogger(clazz);
    }

    @Nullable
    private static URI configUriForDirectory(File directory, EnvironmentType env) {
        String logConfig = configNameForEnvironment(env);
        File configFile = new File(directory, logConfig);
        if (!configFile.exists()) {
            logger.debug("Source {} not found", logConfig);
            return null;
        }
        return configFile.toURI();
    }

    @Nullable
    private static URI configUriForClassPath(EnvironmentType env) {
        String logConfig = configNameForEnvironment(env);
        URL configUrl = LoggingInitializer.class.getResource(logConfig);
        if (configUrl == null) {
            logger.debug("Source {} not found", logConfig);
            return null;
        }
        try {
            return configUrl.toURI();
        } catch (URISyntaxException e) {
            logger.error("Malformed configuration url {}", configUrl, e);
            return null;
        }
    }

    @Nonnull
    private static String configNameForEnvironment(EnvironmentType env) {
        return "/log4j2-" + env.name().toLowerCase() + ".xml";
    }
}
