package ru.yandex.direct.logging;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.layout.PatternLayout;
import org.apache.logging.log4j.core.pattern.ConverterKeys;
import org.apache.logging.log4j.core.pattern.LogEventPatternConverter;
import org.apache.logging.log4j.core.pattern.PatternFormatter;

import ru.yandex.direct.utils.text.StringModifier;

import static ru.yandex.direct.logging.LoggingMaskingConverter.NAME;

/**
 * Если в лог утекают токены или пароли - это нехорошо.
 * Плагин добавляет к Log4j паттерн %MASKED{...}, к данным применяется набор регулярных выражений
 * и если находится приватная информация - оставляем 3 символа в начале, 3 в конце, а остальное заменяем звёздочками
 */
@Plugin(name = "LoggingMaskingConverter", category = "Converter")
@ConverterKeys({NAME})
public class LoggingMaskingConverter extends LogEventPatternConverter {
    public static final String NAME = "MASKED";

    private final StringModifier stringModifier;
    private final List<PatternFormatter> formatters;

    private LoggingMaskingConverter(Configuration config, String[] options) {
        super(NAME, NAME);

        formatters = PatternLayout.createPatternParser(config).parse(options[0]);
        stringModifier = new StringModifier.Builder()
                .withRegexpReplaceAllRule(
                        "(token=)([a-zA-Z0-9\\-\\_\\.\\~\\+\\/]{10,})",
                        m -> m.group(1) + mask(m.group(2))
                )
                .withRegexpReplaceAllRule(
                        "(password=)([a-zA-Z0-9\\-\\_\\.\\~\\+\\/]{4,})",
                        m -> m.group(1) + maskPassword(m.group(2))
                )
                .build();
    }

    public static LoggingMaskingConverter newInstance(final Configuration config, final String[] options) {
        return new LoggingMaskingConverter(config, options);
    }

    @Override
    public void format(LogEvent logEvent, StringBuilder outputMsg) {
        StringBuilder sb = new StringBuilder();
        for (PatternFormatter formatter : formatters) {
            formatter.format(logEvent, sb);
        }
        outputMsg.append(stringModifier.makeReplacements(sb.toString()));
    }

    private static String mask(String str) {
        if (str.length() <= 6) {
            return str;
        } else {
            return str.substring(0, 3)
                    + StringUtils.repeat('*', str.length() - 6)
                    + str.substring(str.length() - 3);
        }
    }

    private static String maskPassword(String str) {
        return StringUtils.repeat('*', str.length());
    }
}
