package ru.yandex.direct.logging

import org.apache.logging.log4j.core.Appender
import org.apache.logging.log4j.core.Core
import org.apache.logging.log4j.core.Filter
import org.apache.logging.log4j.core.LogEvent
import org.apache.logging.log4j.core.appender.AbstractAppender
import org.apache.logging.log4j.core.appender.rewrite.RewritePolicy
import org.apache.logging.log4j.core.appender.rewrite.RewriteAppender
import org.apache.logging.log4j.core.config.AppenderControl
import org.apache.logging.log4j.core.config.AppenderRef
import org.apache.logging.log4j.core.config.Configuration
import org.apache.logging.log4j.core.config.plugins.Plugin
import org.apache.logging.log4j.core.config.plugins.PluginAttribute
import org.apache.logging.log4j.core.config.plugins.PluginConfiguration
import org.apache.logging.log4j.core.config.plugins.PluginElement
import org.apache.logging.log4j.core.config.plugins.PluginFactory
import org.apache.logging.log4j.core.impl.Log4jLogEvent
import org.apache.logging.log4j.message.SimpleMessage
import java.util.concurrent.ConcurrentHashMap

/**
 * Творчески переписанный [RewriteAppender].
 * Добавлена возможность дополнительно логгировать сообщения, превышающие лимит по размеру
 */
@Plugin(
    name = "RewriteOverflowing",
    category = Core.CATEGORY_NAME,
    elementType = Appender.ELEMENT_TYPE,
    printObject = true,
)
class RewriteOverflowingAppender(
    private val configuration: Configuration,
    name: String,
    private val overflowAppenderName: String?,
    private val appenderRefs: Array<AppenderRef>,
    private val rewritePolicy: RewritePolicy?,
    filter: Filter?,
) : AbstractAppender(name, filter, null, true, null) {

    private var overflowControl: AppenderControl? = null
    private val appenders: MutableMap<String, AppenderControl> = ConcurrentHashMap()

    override fun start() {
        if (overflowAppenderName != null) {
            val appender: Appender = configuration.getAppender(overflowAppenderName)
                ?: throw IllegalStateException("Appender $overflowAppenderName cannot be located")
            overflowControl = AppenderControl(appender, null, null)
        }

        appenderRefs.forEach { ref ->
            val appender: Appender? = configuration.getAppender(ref.ref)
            if (appender == null) {
                LOGGER.error("Appender $ref cannot be located. Reference ignored")
            } else {
                val filter = (appender as? AbstractAppender)?.filter
                appenders[ref.ref] = AppenderControl(appender, ref.level, filter)
            }
        }

        super.start()
    }

    override fun append(event: LogEvent) {
        if (rewritePolicy != null) {
            appendInternal(rewritePolicy.rewrite(event))
        } else {
            appendInternal(event)
        }
    }

    /**
     * Намеренно логгируем большие сообщения и в другие аппендеры, дополнительный вывод в overflow аппендер нужен только
     * для дебага. Обнаруживать потери сообщений все еще хотим по метрикам unified agent
     */
    private fun appendInternal(event: LogEvent) {
        val overflowControl = overflowControl
        if (overflowControl != null) {
            val formattedMessage = event.message.formattedMessage
            val messageSizeBytes = formattedMessage.toByteArray().size

            if (messageSizeBytes > DEFAULT_MAX_MESSAGE_SIZE_BYTES) {
                val overflowEvent = Log4jLogEvent.Builder(event)
                    .setLoggerName(RewriteOverflowingAppender::class.qualifiedName)
                    .setMessage(SimpleMessage("Too long message from logger ${event.loggerName}: $formattedMessage"))
                    .build()

                overflowControl.callAppender(overflowEvent)
            }
        }

        appenders.values.forEach { appender ->
            appender.callAppender(event)
        }
    }

    companion object {
        // Совпадает с GrpcMaxMessageSize
        // Сообщение попадёт в overflow только если оно будет откинуто UnifiedAgentAppender
        private const val DEFAULT_MAX_MESSAGE_SIZE_BYTES = 1 shl 21 // 2mb

        @JvmStatic
        @PluginFactory
        fun createAppender(
            @PluginConfiguration configuration: Configuration,
            @PluginAttribute("name") name: String,
            @PluginAttribute("overflowAppender") overflowAppenderName: String?,
            @PluginElement("AppenderRef") appenderRefs: Array<AppenderRef>,
            @PluginElement("RewritePolicy") rewritePolicy: RewritePolicy?,
            @PluginElement("Filter") filter: Filter?,
        ) = RewriteOverflowingAppender(
            configuration,
            name,
            overflowAppenderName,
            appenderRefs,
            rewritePolicy,
            filter,
        )
    }
}
