package ru.yandex.direct.logging;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.Marker;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.Logger;
import org.apache.logging.log4j.core.config.Node;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.filter.AbstractFilter;
import org.apache.logging.log4j.message.Message;

/**
 * Фильтр для накапливания статистики о количестве залоггированных сообщений на разных уровнях.
 */
@Plugin(name = "StatsFilter", category = Node.CATEGORY, elementType = Filter.ELEMENT_TYPE, printObject = true)
public class StatsFilter extends AbstractFilter {
    private static final ConcurrentHashMap<String, ConcurrentHashMap<Level, AtomicLong>> STATS =
            new ConcurrentHashMap<>();
    private static final String DEFAULT_STAT_NAME = "log";

    private final String statName;

    /**
     * @param statName - идентификатор, по которому будет собираться статистика, по умолчанию "log"
     */
    private StatsFilter(String statName) {
        super(Result.NEUTRAL, Result.NEUTRAL);
        this.statName = statName != null ? statName : DEFAULT_STAT_NAME;
    }

    @Override
    public Result filter(Logger logger, Level level, Marker marker, String msg,
                         Object... params) {
        return filter(level);
    }

    @Override
    public Result filter(Logger logger, Level level, Marker marker, Object msg,
                         Throwable t) {
        return filter(level);
    }

    @Override
    public Result filter(Logger logger, Level level, Marker marker, Message msg,
                         Throwable t) {
        return filter(level);
    }

    @Override
    public Result filter(LogEvent event) {
        return filter(event.getLevel());
    }

    private Result filter(Level level) {
        STATS.computeIfAbsent(statName, s -> new ConcurrentHashMap<>())
                .computeIfAbsent(level, l -> new AtomicLong())
                .incrementAndGet();
        return Result.NEUTRAL;
    }

    @Override
    public String toString() {
        return "StatFilter(" + statName + ")";
    }

    /**
     * Создание фильтра
     *
     * @param statName - имя, на которое будет записываться статистика
     */
    @PluginFactory
    public static StatsFilter createFilter(
            @PluginAttribute("statName") String statName) {
        return new StatsFilter(statName);
    }

    /**
     * Получить копию накопленной статистики
     */
    public static List<StatItem> getStats() {
        List<StatItem> ret = new ArrayList<>();
        STATS.forEach((statName, map) ->
                map.forEach((level, cnt) ->
                        ret.add(new StatItem(statName, level, cnt.get())))
        );
        return ret;
    }

    /**
     * элемент статистики о количестве залогированных сообщений
     */
    public static class StatItem {
        private final String statName;
        private final Level level;
        private final long count;

        private StatItem(String statName, Level level, long count) {
            this.statName = statName;
            this.level = level;
            this.count = count;
        }

        /**
         * идентификатор, на который собираласть статистика, определяется при создании StatsFilter
         */
        public String statName() {
            return statName;
        }

        /**
         * уровень логирования
         */
        public Level level() {
            return level;
        }

        /**
         * количество залогированных сообщений
         */
        public long count() {
            return count;
        }
    }
}
