package ru.yandex.direct.version;

import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.library.svnversion.VcsVersion;

/**
 * Информация о версии программы, собранной из Аркадии.
 * Использует аркадийную библиотеку svnversion, работает только с Svn.
 * Может обманывать(а также возвращать нули/unknown), сильно доверять нельзя.
 */
public class DirectVersion {


    private static class Holder {
        static final VcsVersion VCS = new VcsVersion(DirectVersion.class);
        static final Version INSTANCE = new Version(VCS);
    }

    public static final String UNKNOWN_VERSION = "unknown";

    public static final String BUILD_VERSION_PROPERTY = "build.version";

    private DirectVersion() {
    }

    public static VcsVersion getVcsVersion() {
        return Holder.VCS;
    }

    /**
     * Предположительная версия пакета Директа:
     * <li>для пакетов собранных через sandbox-ya-package имеет формат 1-9601582.1 или 1-9601582.9601831-1
     * <li>для пакетов собранных из CI имеет формат 63 или 63.1
     * <li>если версию определить не получилось, {@code DirectVersion.UNKNOWN_VERSION}
     */
    public static String getVersion() {
        return Holder.INSTANCE.version;
    }

    /**
     * <li>для пакетов собранных через sandbox-ya-package &mdash; транковая ревизия, из которой собран пакет, или
     * от которой отведён релизный бранч
     * <li>для пакетов собранных из CI &mdash; номер релиза
     * <li>если версию определить не получилось, {@code 0}
     */
    public static int getMajorVersion() {
        return Holder.INSTANCE.major;
    }

    /**
     * <li>для пакетов собранных через sandbox-ya-package &mdash; ревизия последнего коммита в релизной ветке
     * (совпадает с {@code DirectVersion::getMajorVersion} для релизов из транка)
     * <li>для пакетов собранных из CI &mdash; минорная версия релиза, либо {@code 0}
     * <li>если версию определить не получилось, {@code -1}
     */
    public static int getMinorVersion() {
        return Holder.INSTANCE.minor;
    }

    static class Version {
        private static final Logger logger = LoggerFactory.getLogger(Version.class);

        private static final Pattern TRUNK_PATTERN =
                Pattern.compile("^.*://([^/]+@)?arcadia.yandex.ru/arc/trunk/arcadia$");
        private static final Pattern BRANCH_PATTERN =
                Pattern.compile("^.*://([^/]+@)?arcadia.yandex.ru/arc/branches/direct/release/[\\w-]+/(\\d+)/arcadia$");
        private static final Pattern BUILD_VERSION_PATTERN = Pattern.compile("^(\\d+)(?:\\.(\\d+))?$");

        int major = 0;
        int minor = 0;
        String version = UNKNOWN_VERSION;

        Version(VcsVersion vcs) {
            this(vcs.getProgramSvnRevision(), vcs.getArcadiaSourceUrl());
        }

        Version(int revision, String sourceUrl) {
            // Первым приоритетом пытаемся использовать версию переданную при запуске приложения
            String buildVersion = System.getProperty(BUILD_VERSION_PROPERTY);
            if (StringUtils.isNotEmpty(buildVersion)) {
                fillFromCiVersion(buildVersion);
            } else {
                fillVersionFromSvnRevision(revision, sourceUrl);
            }
        }

        private void fillVersionFromSvnRevision(int revision, String sourceUrl) {
            minor = revision;
            if (TRUNK_PATTERN.matcher(sourceUrl).matches()) {
                major = minor;
            } else {
                Matcher matcher = BRANCH_PATTERN.matcher(sourceUrl);
                if (!matcher.matches()) {
                    logger.error("Can't determine program version: {}", sourceUrl);
                    version = UNKNOWN_VERSION;
                    return;
                }

                try {
                    major = Integer.parseInt(matcher.group(2));
                } catch (Exception e) {
                    logger.error("Can't determine program version", e);
                    version = UNKNOWN_VERSION;
                    return;
                }
            }

            if (major == minor) {
                version = "1." + major + "-1";
            } else {
                version = "1." + major + "." + minor + "-1";
            }
        }

        private void fillFromCiVersion(String buildVersion) {
            logger.info("build.version = {}", buildVersion);

            Matcher buildVersionMatcher = BUILD_VERSION_PATTERN.matcher(buildVersion);
            if (!buildVersionMatcher.matches()) {
                logger.error("Unexpected build.version: {}", buildVersion);
                version = UNKNOWN_VERSION;
                return;
            }

            major = Integer.parseInt(buildVersionMatcher.group(1));
            try {
                minor = Optional.ofNullable(buildVersionMatcher.group(2))
                        .map(Integer::parseInt)
                        .orElse(0);
            } catch (NumberFormatException e) {
                logger.error("Unexpected build.version, failed to parse minor version: {}", buildVersion);
                version = UNKNOWN_VERSION;
                return;
            }

            if (minor == 0) {
                version = String.valueOf(major);
            } else {
                version = major + "." + minor;
            }
        }
    }
}
