package ru.yandex.direct.core.entity.aggregatedstatuses;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;

import static ru.yandex.direct.core.entity.aggregatedstatuses.SelfStatus.status;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class AggregatedStatusBaseData<T> {
    public static final String STATES = "sts";
    public static final String STATUS = "s";
    public static final String REASONS = "r";
    public static final String COUNTERS = "cnts";
    public static final String REJECT_REASONS = "rr";

    private final Collection<T> states;

    @JsonIgnore
    private SelfStatus selfStatus;

    @JsonIgnore
    private Boolean isObsolete;

    @JsonIgnore
    private CampaignsPlatform platform;

    @JsonCreator
    public AggregatedStatusBaseData(@JsonProperty(STATES) Collection<T> states,
                                    @JsonProperty(STATUS) GdSelfStatusEnum status,
                                    @JsonProperty(REASONS) List<GdSelfStatusReason> reasons,
                                    @JsonProperty(REJECT_REASONS) Map<ModerationDiagType, Set<Long>> rejectReasons) {
        this.states = nvl(states, Collections.emptySet());
        this.selfStatus = new SelfStatus(status, reasons, rejectReasons);
    }

    public AggregatedStatusBaseData(Collection<T> states,
                                    GdSelfStatusEnum status,
                                    List<GdSelfStatusReason> reasons) {
        this.states = nvl(states, Collections.emptySet());
        this.selfStatus = status(status, reasons);
    }

    public AggregatedStatusBaseData(Collection<T> states,
                                    GdSelfStatusEnum status,
                                    GdSelfStatusReason reason) {
        this.states = nvl(states, Collections.emptySet());
        this.selfStatus = status(status, reason);
    }

    public AggregatedStatusBaseData(Collection<T> states,
                                    GdSelfStatusEnum status) {
        this.states = nvl(states, Collections.emptySet());
        this.selfStatus = status(status);
    }

    public AggregatedStatusBaseData(Collection<T> states,
                                    SelfStatus status) {
        this.states = nvl(states, Collections.emptySet());
        this.selfStatus = status;
    }

    @JsonGetter(STATES)
    public Collection<T> getStatesUnsafe() {
        return states.isEmpty() ? null : states;
    }

    @JsonIgnore
    public Collection<T> getStates() {
        return states;
    }

    @JsonIgnore
    public Optional<GdSelfStatusEnum> getStatus() {
        return selfStatus != null ? Optional.ofNullable(selfStatus.getStatus()) : Optional.empty();
    }

    @JsonIgnore
    public List<GdSelfStatusReason> getReasons() {
        return selfStatus != null && selfStatus.getReasons() != null
                ? selfStatus.getReasons() : Collections.emptyList();
    }

    @JsonIgnore
    public Map<ModerationDiagType, Set<Long>> getRejectReasons() {
        return selfStatus != null && selfStatus.getRejectReasons() != null
                ? selfStatus.getRejectReasons() : Collections.emptyMap();
    }

    @JsonGetter(STATUS)
    public GdSelfStatusEnum getStatusUnsafe() {
        return selfStatus != null ? selfStatus.getStatus() : null;
    }

    @JsonGetter(REASONS)
    public List<GdSelfStatusReason> getReasonsUnsafe() {
        return selfStatus != null ? selfStatus.getReasons() : null;
    }

    @JsonGetter(REJECT_REASONS)
    public Map<ModerationDiagType, Set<Long>> getRejectReasonsUnsafe() {
        return selfStatus != null ? selfStatus.getRejectReasons() : null;
    }

    @JsonIgnore
    public Optional<SelfStatus> getSelfStatusObject() {
        return Optional.ofNullable(selfStatus);
    }

    @JsonIgnore
    public void updateSelfStatus(SelfStatus selfStatus) {
        this.selfStatus = selfStatus;
    }

    @JsonIgnore
    public Boolean getIsObsolete() {
        return isObsolete;
    }

    @JsonIgnore
    public void setIsObsolete(Boolean isObsolete) {
        this.isObsolete = isObsolete;
    }

    @JsonIgnore
    public CampaignsPlatform getPlatform() {
        return platform;
    }

    @JsonIgnore
    public void setPlatform(CampaignsPlatform platform) {
        this.platform = platform;
    }

    @Override
    public String toString() {
        return toJson(this);
    }
}
