package ru.yandex.direct.core.entity.aggregatedstatuses;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;

@ParametersAreNonnullByDefault
public class SelfStatus {
    private final GdSelfStatusEnum status;
    @Nullable
    private final List<GdSelfStatusReason> reasons;
    @Nullable
    private final Map<ModerationDiagType, Set<Long>> rejectReasons;

    public SelfStatus(GdSelfStatusEnum status, GdSelfStatusReason reason,
                      Map<ModerationDiagType, Set<Long>> rejectReasons) {
        this(status, Collections.singletonList(reason), rejectReasons);
    }

    public SelfStatus(GdSelfStatusEnum status, GdSelfStatusReason reason) {
        this(status, Collections.singletonList(reason), null);
    }

    public SelfStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons) {
        this(status, reasons, null);
    }

    public SelfStatus(GdSelfStatusEnum status) {
        this(status, (List<GdSelfStatusReason>) null, null);
    }

    public SelfStatus(GdSelfStatusEnum status,
                      @Nullable List<GdSelfStatusReason> reasons,
                      @Nullable Map<ModerationDiagType, Set<Long>> rejectReasons) {
        this.status = status;
        this.reasons = reasons != null
                ? reasons.stream().sorted(new GdSelfStatusReasonComparator()).collect(Collectors.toUnmodifiableList())
                : null;
        this.rejectReasons = rejectReasons;
    }

    public GdSelfStatusEnum getStatus() {
        return status;
    }

    public @Nullable
    List<GdSelfStatusReason> getReasons() {
        return reasons;
    }

    public @Nullable
    Map<ModerationDiagType, Set<Long>> getRejectReasons() {
        return rejectReasons;
    }

    public static SelfStatus status(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                    Map<ModerationDiagType, Set<Long>> rejectReasons) {
        return new SelfStatus(status, reasons, rejectReasons);
    }

    public static SelfStatus status(GdSelfStatusEnum status, GdSelfStatusReason reason,
                                    Map<ModerationDiagType, Set<Long>> rejectReasons) {
        return new SelfStatus(status, reason, rejectReasons);
    }

    public static SelfStatus status(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons) {
        return new SelfStatus(status, reasons);
    }

    public static SelfStatus status(GdSelfStatusEnum status, GdSelfStatusReason reason) {
        return new SelfStatus(status, reason);
    }

    public static SelfStatus status(GdSelfStatusEnum status) {
        return new SelfStatus(status);
    }

    public static SelfStatus copySetStatus(SelfStatus selfStatus, GdSelfStatusEnum status) {
        var reasons = selfStatus.reasons != null ? List.copyOf(selfStatus.reasons) : null;
        var rejectReasons = selfStatus.rejectReasons != null ? Map.copyOf(selfStatus.rejectReasons) : null;
        return new SelfStatus(status, reasons, rejectReasons);
    }

    public static SelfStatus copyAddReason(SelfStatus status, GdSelfStatusReason reason) {
        return copyAddReasons(status, Collections.singletonList(reason));
    }

    public static SelfStatus copyAddReasons(SelfStatus status, List<GdSelfStatusReason> reasons) {
        var rejectReasons = status.rejectReasons != null ? Map.copyOf(status.rejectReasons) : null;
        return new SelfStatus(status.getStatus(), mergeReasons(reasons, status.getReasons()), rejectReasons);
    }

    public static SelfStatus copyAddSecondaryReason(SelfStatus status, GdSelfStatusReason reason) {
        return copyAddSecondaryReasons(status, Collections.singletonList(reason));
    }

    public static SelfStatus copyAddSecondaryReasons(SelfStatus status, List<GdSelfStatusReason> reasons) {
        var rejectReasons = status.rejectReasons != null ? Map.copyOf(status.rejectReasons) : null;
        return new SelfStatus(status.getStatus(), mergeReasons(status.getReasons(), reasons), rejectReasons);
    }

    private static List<GdSelfStatusReason> mergeReasons(@Nullable List<GdSelfStatusReason> firstReasons,
                                                         @Nullable List<GdSelfStatusReason> secondReasons) {
        if (secondReasons == null) {
            return firstReasons != null ? List.copyOf(firstReasons) : null;
        }
        if (firstReasons == null) {
            return List.copyOf(secondReasons);
        }
        List<GdSelfStatusReason> reasons = new ArrayList<>(firstReasons);
        reasons.addAll(secondReasons);
        return reasons;
    }

    public static SelfStatus copyAddRejectReasons(SelfStatus status, Map<ModerationDiagType, Set<Long>> rejectReasons) {
        var reasons = status.reasons != null ? List.copyOf(status.reasons) : null;
        if (rejectReasons == null) {
            return new SelfStatus(status.getStatus(), reasons,
                    status.rejectReasons != null ? Map.copyOf(status.rejectReasons) : null);
        }
        if (status.getRejectReasons() == null) {
            return new SelfStatus(status.getStatus(), reasons, Map.copyOf(rejectReasons));
        }
        Map<ModerationDiagType, Set<Long>> newRejectReasons = new EnumMap<>(ModerationDiagType.class);
        newRejectReasons.putAll(status.getRejectReasons());
        for (var entry : rejectReasons.entrySet()) {
            newRejectReasons.computeIfAbsent(entry.getKey(), k -> new HashSet<>()).addAll(entry.getValue());
        }
        return new SelfStatus(status.getStatus(), reasons, newRejectReasons);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SelfStatus that = (SelfStatus) o;
        return getStatus() == that.getStatus() &&
                Objects.equals(getReasons(), that.getReasons()) &&
                Objects.equals(getRejectReasons(), that.getRejectReasons());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getStatus(), getReasons(), getRejectReasons());
    }

    @Override
    public String toString() {
        return "SelfStatus{" +
                "status=" + status +
                ", reasons=" + reasons +
                ", rejectReasons=" + rejectReasons +
                '}';
    }
}
