package ru.yandex.direct.core.entity.aggregatedstatuses.adgroup;

import java.util.Collection;
import java.util.EnumMap;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AdStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.KeywordStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.RetargetingStatesEnum;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.JsonUtils.toDeterministicJson;

@JsonInclude(JsonInclude.Include.NON_NULL)
public class AdGroupCounters {
    @JsonProperty("ads")
    private int adsTotal;

    @JsonProperty("kws")
    private int keywordsTotal;

    @JsonProperty("rets")
    private int retargetingsTotal;

    @JsonProperty("b_sts")
    private final Map<AdStatesEnum, Integer> adStates;
    @JsonProperty("b_s")
    private final Map<GdSelfStatusEnum, Integer> adStatuses;

    @JsonProperty("kw_sts")
    private final Map<KeywordStatesEnum, Integer> keywordStates;
    @JsonProperty("kw_s")
    private final Map<GdSelfStatusEnum, Integer> keywordStatuses;

    @JsonProperty("ret_sts")
    private final Map<RetargetingStatesEnum, Integer> retargetingStates;
    @JsonProperty("ret_s")
    private final Map<GdSelfStatusEnum, Integer> retargetingStatuses;

    // какие-то кастомные счетчики можно добавить отдельным полем

    public AdGroupCounters(int adsTotal, int keywordsTotal, int retargetingsTotal,
                           Map<GdSelfStatusEnum, Integer> adStatuses,
                           Map<AdStatesEnum, Integer> adStates,
                           Map<GdSelfStatusEnum, Integer> keywordStatuses,
                           Map<KeywordStatesEnum, Integer> keywordStates,
                           Map<GdSelfStatusEnum, Integer> retargetingStatuses,
                           Map<RetargetingStatesEnum, Integer> retargetingStates) {
        this.adsTotal = adsTotal;
        this.keywordsTotal = keywordsTotal;
        this.retargetingsTotal = retargetingsTotal;
        this.adStatuses = nvl(adStatuses, new EnumMap<>(GdSelfStatusEnum.class));
        this.adStates = nvl(adStates, new EnumMap<>(AdStatesEnum.class));
        this.keywordStatuses = nvl(keywordStatuses, new EnumMap<>(GdSelfStatusEnum.class));
        this.keywordStates = nvl(keywordStates, new EnumMap<>(KeywordStatesEnum.class));
        this.retargetingStatuses = nvl(retargetingStatuses, new EnumMap<>(GdSelfStatusEnum.class));
        this.retargetingStates = nvl(retargetingStates, new EnumMap<>(RetargetingStatesEnum.class));
    }

    public AdGroupCounters() {
        adsTotal = 0;
        keywordsTotal = 0;
        retargetingsTotal = 0;
        adStatuses = new EnumMap<>(GdSelfStatusEnum.class);
        adStates = new EnumMap<>(AdStatesEnum.class);
        keywordStatuses = new EnumMap<>(GdSelfStatusEnum.class);
        keywordStates = new EnumMap<>(KeywordStatesEnum.class);
        retargetingStatuses = new EnumMap<>(GdSelfStatusEnum.class);
        retargetingStates = new EnumMap<>(RetargetingStatesEnum.class);
    }

    @JsonIgnore
    public Map<GdSelfStatusEnum, Integer> getAdStatuses() {
        return adStatuses;
    }

    @JsonIgnore
    public Map<AdStatesEnum, Integer> getAdStates() {
        return adStates;
    }

    @JsonIgnore
    public Map<GdSelfStatusEnum, Integer> getKeywordStatuses() {
        return keywordStatuses;
    }

    @JsonIgnore
    public Map<GdSelfStatusEnum, Integer> getRetargetingStatuses() {
        return retargetingStatuses;
    }

    @JsonIgnore
    public Map<RetargetingStatesEnum, Integer> getRetargetingStates() {
        return retargetingStates;
    }


    @JsonGetter("b_s")
    public Map<GdSelfStatusEnum, Integer> getAdStatusesUnsafe() {
        return nullIfEmpty(adStatuses);
    }

    @JsonGetter("b_sts")
    public Map<AdStatesEnum, Integer> getAdStatesUnsafe() {
        return nullIfEmpty(adStates);
    }

    @JsonGetter("kw_s")
    public Map<GdSelfStatusEnum, Integer> getKeywordStatusesUnsafe() {
        return nullIfEmpty(keywordStatuses);
    }

    @JsonGetter("kw_sts")
    public Map<KeywordStatesEnum, Integer> getKeywordStatesUnsafe() {
        return nullIfEmpty(keywordStates);
    }

    @JsonGetter("ret_s")
    public Map<GdSelfStatusEnum, Integer> getRetargetingStatusesUnsafe() {
        return nullIfEmpty(retargetingStatuses);
    }

    @JsonGetter("ret_sts")
    public Map<RetargetingStatesEnum, Integer> getRetargetingStatesUnsafe() {
        return nullIfEmpty(retargetingStates);
    }

    public int getAdsTotal() {
        return adsTotal;
    }

    public int getKeywordsTotal() {
        return keywordsTotal;
    }

    public int getRetargetingsTotal() {
        return retargetingsTotal;
    }

    public void incrAdsCount() {
        adsTotal++;
    }

    public void incrKeywordsCount() {
        keywordsTotal++;
    }

    public void incrRetargetingsCount() {
        retargetingsTotal++;
    }

    public void incrAd(AdStatesEnum state) {
        adStates.compute(state, (s, c) -> c == null ? 1 : c + 1);
    }

    public void incrAd(Collection<AdStatesEnum> states) {
        for (AdStatesEnum state : states) {
            incrAd(state);
        }
    }

    public void incrAd(GdSelfStatusEnum status) {
        adStatuses.compute(status, (s, c) -> c == null ? 1 : c + 1);
    }

    public void incrKeyword(KeywordStatesEnum state) {
        keywordStates.compute(state, (s, c) -> c == null ? 1 : c + 1);
    }

    public void incrKeyword(Collection<KeywordStatesEnum> states) {
        for (KeywordStatesEnum state : states) {
            incrKeyword(state);
        }
    }

    public void incrKeyword(GdSelfStatusEnum status) {
        keywordStatuses.compute(status, (s, c) -> c == null ? 1 : c + 1);
    }

    public void incrRetargeting(RetargetingStatesEnum state) {
        retargetingStates.compute(state, (s, c) -> c == null ? 1 : c + 1);
    }

    public void incrRetargeting(Collection<RetargetingStatesEnum> states) {
        for (RetargetingStatesEnum state : states) {
            incrRetargeting(state);
        }
    }

    public void incrRetargeting(GdSelfStatusEnum status) {
        retargetingStatuses.compute(status, (s, c) -> c == null ? 1 : c + 1);
    }

    private <T extends Map> T nullIfEmpty(T map) {
        return map.isEmpty() ? null : map;
    }

    @Override
    public String toString() {
        return toDeterministicJson(this);
    }
}
