package ru.yandex.direct.core.entity.aggregatedstatuses.campaign;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.campaign.model.CampaignFilterStatus;

import static org.apache.commons.collections4.CollectionUtils.containsAny;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.CAMPAIGN_ADD_MONEY;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.CAMPAIGN_HAS_NO_ADS_ELIGIBLE_FOR_SERVING;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.CAMPAIGN_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.CAMPAIGN_WAIT_PAYMENT;

public class CampaignStatusCalculator {

    private static final Logger logger = LoggerFactory.getLogger(CampaignStatusCalculator.class);

    private static final Set<GdSelfStatusEnum> MODERATION_SET =
            Set.of(GdSelfStatusEnum.DRAFT, GdSelfStatusEnum.STOP_CRIT);

    private static final Set<CampaignStatesEnum> MODERATION_REJECTED_STATES_SET =
            Set.of(CampaignStatesEnum.PAYED, CampaignStatesEnum.NO_MONEY, CampaignStatesEnum.AWAIT_PAYMENT);

    public static CampaignFilterStatus convertToFilterStatus(AggregatedStatusCampaignData aggregatedStatus) {
        if (aggregatedStatus == null || aggregatedStatus.getStatus().isEmpty()) {
            return CampaignFilterStatus.DRAFT;
        }

        var status = aggregatedStatus.getStatus().get();

        if (status == GdSelfStatusEnum.RUN_WARN) {
            return CampaignFilterStatus.RUN_WARN;
        }

        var reasons = ListUtils.emptyIfNull(aggregatedStatus.getReasons());
        var states = CollectionUtils.emptyIfNull(aggregatedStatus.getStates());

        // Архивные
        if (isArchivedPrimaryStatus(status, reasons, states)) {
            return CampaignFilterStatus.ARCHIVED;
        }
        // На модерации
        else if (isModerationPrimaryStatus(status, reasons, states)) {
            return CampaignFilterStatus.MODERATION;
        }
        // Отклоненные
        else if (isModerationDeniedPrimaryStatus(status, reasons, states)) {
            return CampaignFilterStatus.MODERATION_DENIED;
        }
        // Запущенные
        else if (isTemporarilyPausedPrimaryStatus(status, reasons, states)) {
            return CampaignFilterStatus.TEMPORARILY_PAUSED;
        }
        // Остановленные
        else if (isStoppedPrimaryStatus(status, reasons, states)) {
            return CampaignFilterStatus.STOPPED;
        }
        // Черновики
        else if (isDraftPrimaryStatus(status, reasons, states)) {
            return CampaignFilterStatus.DRAFT;
        }
        // Запущенные
        else if (isActivePrimaryStatus(status, reasons, states)) {
            return CampaignFilterStatus.ACTIVE;
        }

        logger.warn("Can't determine the filter status for {}", aggregatedStatus);
        return CampaignFilterStatus.ACTIVE;
    }

    // Архивные
    public static boolean isArchivedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                  Collection<CampaignStatesEnum> states) {
        return Objects.equals(status, GdSelfStatusEnum.ARCHIVED);
    }

    // На модерации
    public static boolean isModerationPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                    Collection<CampaignStatesEnum> states) {
        return GdSelfStatusEnum.ON_MODERATION.equals(status)
                || (MODERATION_SET.contains(status) && reasons.contains(CAMPAIGN_ON_MODERATION));
    }

    // Отклоненные
    public static boolean isModerationDeniedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                          Collection<CampaignStatesEnum> states) {
        return Objects.equals(status, GdSelfStatusEnum.STOP_CRIT)
                && (containsAny(states, MODERATION_REJECTED_STATES_SET))
                && reasons.contains(CAMPAIGN_HAS_NO_ADS_ELIGIBLE_FOR_SERVING);
    }

    // Запущенные: Данный кейс пока отображается в 'Остановленные', т.к. при прохождении кампанией finish_time ее не
    // корректно отображать в Запущенных -> DIRECT-122712
    public static boolean isNoMoneyPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                 Collection<CampaignStatesEnum> states) {
        return Objects.equals(status, GdSelfStatusEnum.STOP_CRIT)
                && states.contains(CampaignStatesEnum.NO_MONEY)
                && containsAny(reasons, Set.of(CAMPAIGN_WAIT_PAYMENT, CAMPAIGN_ADD_MONEY));
    }

    // Запущенные
    public static boolean isTemporarilyPausedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                           Collection<CampaignStatesEnum> states) {
        return GdSelfStatusEnum.allPause().contains(status);
    }

    // Остановленные
    public static boolean isStoppedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                 Collection<CampaignStatesEnum> states) {
        return GdSelfStatusEnum.allStop().contains(status) && status != GdSelfStatusEnum.STOP_PROCESSING;
    }

    // Черновики
    public static boolean isDraftPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                               Collection<CampaignStatesEnum> states) {
        return Objects.equals(status, GdSelfStatusEnum.DRAFT);
    }

    // Запущенные
    public static boolean isActivePrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                Collection<CampaignStatesEnum> states) {
        return GdSelfStatusEnum.allRun().contains(status) || status == GdSelfStatusEnum.STOP_PROCESSING;
    }

}
