package ru.yandex.direct.core.entity.autobudget.model;

import java.util.Arrays;
import java.util.BitSet;
import java.util.EnumSet;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Все варианты автобюджетных проблем вместе с соответствующими номерами битов в битовой маске,
 * которую присылает нам БК, и которую мы храним в БД
 */
@ParametersAreNonnullByDefault
public enum AutobudgetHourlyProblem {
    IN_ROTATION(0),
    MAX_BID_REACHED(1),
    MARGINAL_PRICE_REACHED(2),
    UPPER_POSITIONS_REACHED(3),
    ENGINE_MIN_COST_LIMITED(4),
    LIMITED_BY_BALANCE(5),
    NO_BANNERS(6),
    LIMIT_BY_AVG_COST(7),
    WALLET_DAILY_BUDGET_REACHED(8);

    private int bitNumber;

    AutobudgetHourlyProblem(int bitNumber) {
        this.bitNumber = bitNumber;
    }

    /**
     * Переводит битовую маску автобюджетных проблем из формата БД в EnumSet
     *
     * @param bitSet битовая маска из БД
     * @return EnumSet с автобюджетными проблемами, которые выражаются битовой маской
     */
    public static Set<AutobudgetHourlyProblem> bitSetToEnumSet(BitSet bitSet) {
        EnumSet<AutobudgetHourlyProblem> result = EnumSet.noneOf(AutobudgetHourlyProblem.class);
        result.addAll(Arrays.stream(AutobudgetHourlyProblem.values())
                .filter(problem -> bitSet.get(problem.bitNumber))
                .collect(Collectors.toSet())
        );
        return result;
    }

    /**
     * Переводит множество автобюджетных проблем в битовую маску, которую можно потом записать в БД
     *
     * @param problems множество автобюджетных проблем
     * @return битовая маска, описывающая множество автобюджетных проблем
     */
    public static BitSet enumSetToBitSet(Set<AutobudgetHourlyProblem> problems) {
        BitSet result = new BitSet();
        problems.forEach(problem -> result.set(problem.bitNumber));
        return result;
    }
}
