package ru.yandex.direct.core.entity.banner.model;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.banner.model.Age.AGE_18;
import static ru.yandex.direct.core.entity.banner.model.BabyFood.BABY_FOOD_11;

@ParametersAreNonnullByDefault
public class BannerFlags {
    public static final FlagProperty<Boolean> ABORTION = FlagProperty.booleanFlag("abortion");
    public static final FlagProperty<Boolean> MEDICINE = FlagProperty.booleanFlag("medicine");
    public static final FlagProperty<Boolean> MED_SERVICES = FlagProperty.booleanFlag("med_services");
    public static final FlagProperty<Boolean> MED_EQUIPMENT = FlagProperty.booleanFlag("med_equipment");
    public static final FlagProperty<Boolean> PHARMACY = FlagProperty.booleanFlag("pharmacy");
    public static final FlagProperty<Boolean> ALCOHOL = FlagProperty.booleanFlag("alcohol");
    public static final FlagProperty<Boolean> TOBACCO = FlagProperty.booleanFlag("tobacco");
    public static final FlagProperty<Boolean> PLUS18 = FlagProperty.booleanFlag("plus18");
    public static final FlagProperty<Boolean> DIETARYSUPPL = FlagProperty.booleanFlag("dietarysuppl");
    public static final FlagProperty<Boolean> PROJECT_DECLARATION = FlagProperty.booleanFlag("project_declaration");
    public static final FlagProperty<Boolean> TRAGIC = FlagProperty.booleanFlag("tragic");
    public static final FlagProperty<Boolean> ASOCIAL = FlagProperty.booleanFlag("asocial");
    public static final FlagProperty<Boolean> UNFAMILY = FlagProperty.booleanFlag("unfamily");
    public static final FlagProperty<Boolean> PSEUDOWEAPON = FlagProperty.booleanFlag("pseudoweapon");
    public static final FlagProperty<Boolean> FOREX = FlagProperty.booleanFlag("forex");
    public static final FlagProperty<Boolean> EDUCATION = FlagProperty.booleanFlag("education");
    public static final FlagProperty<Boolean> PEOPLE = FlagProperty.booleanFlag("people");
    public static final FlagProperty<Boolean> NOT_ANIMATED = FlagProperty.booleanFlag("not_animated");
    public static final FlagProperty<Boolean> GOODFACE = FlagProperty.booleanFlag("goodface");
    public static final FlagProperty<Boolean> SOCIAL_ADVERTISING = FlagProperty.booleanFlag("social_advertising");

    public static final FlagProperty<BabyFood> BABY_FOOD =
            FlagProperty.enumFlag("baby_food", BABY_FOOD_11, BabyFood::fromTypedValue, BabyFood::getTypedValue);

    public static final FlagProperty<Age> AGE = FlagProperty.enumFlag("age", AGE_18, Age::fromSource, Age::getValue);

    public static final FlagProperty<YaPages> YA_PAGES =
            FlagProperty.enumFlag("ya_pages", YaPages.YA_REMOVE, YaPages::fromTypedValue, YaPages::getTypedValue);

    public static final FlagProperty<Boolean> MINUS_REGION_RU = FlagProperty.booleanFlag("minus_region_ru");
    public static final FlagProperty<Boolean> MINUS_REGION_KZ = FlagProperty.booleanFlag("minus_region_kz");
    public static final FlagProperty<Boolean> MINUS_REGION_UA = FlagProperty.booleanFlag("minus_region_ua");
    public static final FlagProperty<Boolean> MINUS_REGION_RB = FlagProperty.booleanFlag("minus_region_rb");
    public static final FlagProperty<Boolean> MINUS_REGION_TR = FlagProperty.booleanFlag("minus_region_tr");
    public static final FlagProperty<Boolean> MINUS_REGION_UZ = FlagProperty.booleanFlag("minus_region_uz");

    private Map<String, String> flags;

    public BannerFlags() {
        this(new HashMap<>());
    }

    private BannerFlags(Map<String, String> flags) {
        this.flags = Objects.requireNonNull(flags, "flags");
    }

    public static String toSource(@Nullable BannerFlags bannerFlags) {
        if (bannerFlags == null || bannerFlags.getFlags().isEmpty()) {
            return null;
        }

        return EntryStream.of(bannerFlags.getFlags())
                .sortedBy(Map.Entry::getKey)
                .mapKeyValue((key, value) -> value != null ? key + ":" + value : key)
                .joining(",");
    }

    public static BannerFlags fromSource(@Nullable String serializedFlags) {
        if (serializedFlags == null || serializedFlags.isEmpty()) {
            return null;
        }

        Map<String, String> mapFlags = new HashMap<>();
        Arrays.stream(serializedFlags.split(","))
                .forEach(token -> {
                    String[] keyValue = token.split(":");
                    checkState(keyValue.length <= 2, "Can't parse token with more than one ':': %s", token);
                    mapFlags.put(keyValue[0], keyValue.length > 1 ? keyValue[1] : null);
                });

        return new BannerFlags(mapFlags);
    }

    /**
     * Возвращает значение выставленного флага.
     * В случае флага со значением может вернуть null
     */
    @Nullable
    public <T> T get(FlagProperty<T> flag) {
        return flag.extract(flags);
    }

    /**
     * Сохраняет значение флага во внутреннее представление.
     * with(flag, null) равноценно remove(flag)
     */
    public <T> BannerFlags with(FlagProperty<T> flag, @Nullable T value) {
        flag.store(value, flags);
        return this;
    }

    public <T> void remove(FlagProperty<T> flag) {
        flag.remove(flags);
    }

    public Map<String, String> getFlags() {
        return flags;
    }

    public BannerFlags withFlags(Map<String, String> flags) {
        this.flags = flags;
        return this;
    }

    public BannerFlags withFlagsFromModeration(Map<String, String> flags) {
        checkNotNull(flags);
        fixFlag(flags, BannerFlags.AGE.getKey());
        fixFlag(flags, BannerFlags.BABY_FOOD.getKey());
        this.flags = flags;
        return this;
    }

    /**
     * Исправляет неправильный формат флага с "${flagName}:${flagName}X" на "${flagName}:X"
     * (https://st.yandex-team.ru/MODDEV-1053)
     *
     * @param flags    флаги из ответа модерации
     * @param flagName название флага
     */
    private void fixFlag(Map<String, String> flags, String flagName) {
        String ageFlag = flags.get(flagName);
        if (ageFlag != null) {
            flags.put(flagName, ageFlag.replace(flagName, ""));
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        BannerFlags that = (BannerFlags) o;

        return flags.equals(that.flags);
    }

    @Override
    public int hashCode() {
        return flags.hashCode();
    }
}
