package ru.yandex.direct.core.entity.banner.model;

import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

@ParametersAreNonnullByDefault
public abstract class FlagProperty<T> {
    protected final String key;

    private FlagProperty(String key) {
        this.key = key;
    }

    static FlagProperty<Boolean> booleanFlag(String key) {
        return new FlagProperty.BooleanFlag(key);
    }

    /**
     * Дефолтное значение , для случаев если флаг выставлен, но значение не задано
     */
    static <T extends FlagWithValue> FlagProperty<T> enumFlag(String key, T defaultValue,
                                                              Function<String, T> fromTypedValue,
                                                              Function<T, String> getTypedValue) {
        return new EnumFlag<>(key, fromTypedValue, getTypedValue, defaultValue);
    }

    public String getKey() {
        return key;
    }

    /**
     * удалить значение флага
     */
    public void remove(Map<String, String> map) {
        map.remove(key);
    }

    /**
     * Получить значение флага из мапы
     */
    @Nullable
    public abstract T extract(Map<String, String> map);

    /**
     * При передаче null удалять из мапы
     */
    public abstract void store(@Nullable T value, Map<String, String> map);

    public abstract String getDefaultValue();

    /**
     * Класс для описания булева флага,
     */
    private static class BooleanFlag extends FlagProperty<Boolean> {
        BooleanFlag(String key) {
            super(key);
        }

        @Override
        public Boolean extract(Map<String, String> map) {
            return map.containsKey(key);
        }

        @Override
        public void store(@Nullable Boolean value, Map<String, String> map) {
            if (value != null && value) {
                map.put(key, null);
            } else {
                remove(map);
            }
        }

        @Override
        public String getDefaultValue() {
            return "0";
        }
    }


    /**
     * Класс для описания флага перечислимого типа.
     * Поддерживает значение по уомлчанию, если в базе задан флаг без значения.
     */
    private static class EnumFlag<T extends FlagWithValue> extends FlagProperty<T> {

        private final Function<String, T> fromString;
        private final Function<T, String> toString;
        private T defaultValue;

        EnumFlag(String key, Function<String, T> fromString, Function<T, String> toString, T defaultValue) {
            super(key);
            this.fromString = fromString;
            this.toString = toString;
            this.defaultValue = defaultValue;
        }

        @Override
        public T extract(Map<String, String> map) {
            String value = map.get(key);
            T typedValue = fromString.apply(value);
            if (typedValue == null) {
                typedValue = map.containsKey(key) ? defaultValue : null;
            }

            return typedValue;
        }

        @Override
        public void store(@Nullable T value, Map<String, String> map) {
            if (value != null) {
                map.put(key, toString.apply(value));
            } else {
                map.remove(key);
            }
        }

        @Override
        public String getDefaultValue() {
            return defaultValue.getValue();
        }
    }
}
