package ru.yandex.direct.core.entity.campaign.repository;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Splitter;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignDeviceTargeting;
import ru.yandex.direct.core.entity.campaign.model.CampaignEmailNotification;
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.core.entity.time.model.TimeInterval;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsBroadMatchFlag;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsFairauction;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsAutobudget;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCurrency;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCurrencyconverted;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsPaidByCertificate;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.WalletCampaignsAutopayMode;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toCollection;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import static ru.yandex.direct.libs.timetarget.TimeTargetUtils.defaultTimeTarget;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

public class CampaignMappings {

    private static final Logger logger = LoggerFactory.getLogger(CampaignMappings.class);
    private static final Splitter COMMA_SPLITTER = Splitter.on(',');
    private static final String METRIKA_COUNTERS_DELIMITER = ",";
    public static final String DISABLED_DOMAINS_DELIMITER = ",";
    public static final String DISABLED_IPS_DELIMITER = ",";

    public static CampaignsCurrency currencyCodeToDb(CurrencyCode currencyCode) {
        return currencyCode != null ? CampaignsCurrency.valueOf(currencyCode.name()) : null;
    }

    public static CurrencyCode currencyCodeFromDb(CampaignsCurrency currency) {
        return currency != null ? CurrencyCode.valueOf(currency.name()) : null;
    }

    public static CampaignsArchived archivedToDb(Boolean archived) {
        return archived != null ? (archived ? CampaignsArchived.Yes : CampaignsArchived.No) : null;
    }

    public static Boolean archivedFromDb(CampaignsArchived campaignsArchived) {
        return campaignsArchived != null ? (campaignsArchived == CampaignsArchived.Yes) : null;
    }

    public static CampaignsStatusactive statusActiveToDb(Boolean active) {
        return active != null ? (active ? CampaignsStatusactive.Yes : CampaignsStatusactive.No) : null;
    }

    public static Boolean statusActiveFromDb(CampaignsStatusactive campaignsStatusactive) {
        return campaignsStatusactive != null ? (campaignsStatusactive == CampaignsStatusactive.Yes) : null;
    }

    public static CampaignsStatusshow statusShowToDb(Boolean archived) {
        return archived != null ? (archived ? CampaignsStatusshow.Yes : CampaignsStatusshow.No) : null;
    }

    public static Boolean statusShowFromDb(CampaignsStatusshow campaignsStatusShow) {
        return campaignsStatusShow != null ? (campaignsStatusShow == CampaignsStatusshow.Yes) : null;
    }

    public static CampaignsPaidByCertificate paidByCertificateToDb(Boolean paidByCertificate) {
        if (paidByCertificate == null) {
            return null;
        }
        return paidByCertificate ? CampaignsPaidByCertificate.Yes : CampaignsPaidByCertificate.No;
    }

    public static Boolean paidByCertificateFromDb(CampaignsPaidByCertificate campaignsPaidByCertificate) {
        if (campaignsPaidByCertificate == null) {
            return null;
        }
        return campaignsPaidByCertificate == CampaignsPaidByCertificate.Yes;
    }

    public static CampaignsAutobudget autobudgetToDb(Boolean autobudget) {
        return autobudget != null ? (autobudget ? CampaignsAutobudget.Yes : CampaignsAutobudget.No) : null;
    }

    public static Boolean autobudgetFromDb(CampaignsAutobudget campaignsAutobudget) {
        return campaignsAutobudget != null ? (campaignsAutobudget == CampaignsAutobudget.Yes) : null;
    }

    public static CampaignsStatusbssynced statusBsSyncedToDb(StatusBsSynced status) {
        return status != null ? CampaignsStatusbssynced.valueOf(status.toDbFormat()) : null;
    }

    public static StatusBsSynced statusBsSyncedFromDb(CampaignsStatusbssynced status) {
        return status != null ? StatusBsSynced.valueOfDbFormat(status.getLiteral()) : null;
    }

    public static CampaignsCurrencyconverted currencyConvertedToDb(Boolean converted) {
        return converted != null ? (converted ? CampaignsCurrencyconverted.Yes : CampaignsCurrencyconverted.No) : null;
    }

    public static Boolean currencyConvertedFromDb(CampaignsCurrencyconverted converted) {
        return converted != null ? converted == CampaignsCurrencyconverted.Yes : null;
    }

    public static CampOptionsFairauction fairAuctionToDb(Boolean fairAuction) {
        return fairAuction != null ? (fairAuction ? CampOptionsFairauction.Yes : CampOptionsFairauction.No) : null;
    }

    public static Boolean fairAuctionFromDb(CampOptionsFairauction campaignsArchived) {
        return campaignsArchived != null ? (campaignsArchived == CampOptionsFairauction.Yes) : null;
    }

    public static CampaignsStatusempty statusEmptyToDb(Boolean statusEmpty) {
        return statusEmpty != null ? (statusEmpty ? CampaignsStatusempty.Yes : CampaignsStatusempty.No) : null;
    }

    public static Boolean statusEmptyFromDb(CampaignsStatusempty campaignsStatusempty) {
        return campaignsStatusempty != null ? (campaignsStatusempty == CampaignsStatusempty.Yes) : null;
    }

    public static String strategyDataToDb(StrategyData strategyData) {
        return strategyData != null ? toJson(strategyData) : null;
    }

    public static StrategyData strategyDataFromDb(String strategyData) {
        return strategyData != null ? fromJson(strategyData, StrategyData.class) : null;
    }

    public static String optsToDb(EnumSet<CampaignOpts> optsSet) {
        if (optsSet == null) {
            return null;
        }
        return optsSet.stream()
                .map(String::valueOf)
                .map(String::toLowerCase)
                .collect(Collectors.joining(","));
    }

    public static EnumSet<CampaignOpts> optsFromDb(String opts) {
        if (opts == null) {
            return null;
        }
        if (opts.trim().isEmpty()) {
            return EnumSet.noneOf(CampaignOpts.class);
        }
        Collection<CampaignOpts> optsCollection = new ArrayList<>();
        for (String opt : COMMA_SPLITTER.split(opts)) {
            optsCollection.add(CampaignOpts.valueOf(opt.toUpperCase()));
        }
        return EnumSet.copyOf(optsCollection);
    }

    public static String allowedYndxFrontpageTypesToDb(Set<FrontpageCampaignShowType> optsSet) {
        if (optsSet == null) {
            throw new IllegalArgumentException("optsSet can't be null");
        }
        return optsSet.stream()
                .map(String::valueOf)
                .map(String::toLowerCase)
                .collect(Collectors.joining(","));
    }

    public static Set<FrontpageCampaignShowType> allowedYndxFrontpageTypesFromDb(String opts) {
        if (opts == null) {
            return null;
        }
        if (opts.trim().isEmpty()) {
            return EnumSet.noneOf(FrontpageCampaignShowType.class);
        }
        return StreamEx.of(COMMA_SPLITTER.split(opts).iterator())
                .map(frontpageType -> FrontpageCampaignShowType.valueOf(frontpageType.toUpperCase()))
                .collect(toCollection(() -> EnumSet.noneOf(FrontpageCampaignShowType.class)));
    }

    public static String smsFlagsToDb(EnumSet<SmsFlag> smsFlagSet) {
        if (smsFlagSet == null) {
            return null;
        }
        return smsFlagSet.stream()
                .map(String::valueOf)
                .map(String::toLowerCase)
                .collect(Collectors.joining(","));
    }

    public static EnumSet<SmsFlag> smsFlagsFromDb(String smsFlags) {
        if (smsFlags == null) {
            return null;
        }
        if (smsFlags.trim().isEmpty()) {
            return EnumSet.noneOf(SmsFlag.class);
        }
        Collection<SmsFlag> smsFlagCollection = new ArrayList<>();
        for (String opt : COMMA_SPLITTER.split(smsFlags)) {
            smsFlagCollection.add(SmsFlag.valueOf(opt.toUpperCase()));
        }
        return EnumSet.copyOf(smsFlagCollection);
    }

    public static String emailNotificationsToDb(EnumSet<CampaignEmailNotification> emailNotificationSet) {
        if (emailNotificationSet == null) {
            return null;
        }
        return emailNotificationSet.stream()
                .map(Enum::name)
                .map(String::toLowerCase)
                .collect(Collectors.joining(","));
    }

    public static EnumSet<CampaignEmailNotification> emailNotificationsFromDb(String campaignEmailNotificationFlags) {
        if (campaignEmailNotificationFlags == null) {
            return null;
        }
        if (campaignEmailNotificationFlags.trim().isEmpty()) {
            return EnumSet.noneOf(CampaignEmailNotification.class);
        }
        Collection<CampaignEmailNotification> emailNotificationCollection = new ArrayList<>();
        for (String opt : COMMA_SPLITTER.split(campaignEmailNotificationFlags)) {
            emailNotificationCollection.add(CampaignEmailNotification.valueOf(opt.toUpperCase()));
        }
        return EnumSet.copyOf(emailNotificationCollection);
    }

    public static String deviceTargetingToDb(EnumSet<CampaignDeviceTargeting> deviceTargetingSet) {
        if (deviceTargetingSet == null) {
            return null;
        }
        return deviceTargetingSet.stream()
                .map(String::valueOf)
                .map(String::toLowerCase)
                .collect(Collectors.joining(","));
    }

    public static EnumSet<CampaignDeviceTargeting> deviceTargetingFromDb(String deviceTargeting) {
        if (deviceTargeting == null) {
            return null;
        }
        if (deviceTargeting.trim().isEmpty()) {
            return EnumSet.noneOf(CampaignDeviceTargeting.class);
        }
        Collection<CampaignDeviceTargeting> deviceTargetingCollection = new ArrayList<>();
        for (String device : COMMA_SPLITTER.split(deviceTargeting)) {
            deviceTargetingCollection.add(CampaignDeviceTargeting.valueOf(device.toUpperCase()));
        }
        return EnumSet.copyOf(deviceTargetingCollection);
    }

    public static List<Long> metrikaCountersFromDb(String counters) {
        if (counters == null) {
            return null;
        }
        return Arrays.stream(counters.split(METRIKA_COUNTERS_DELIMITER))
                .filter(m -> !m.isEmpty())
                .map(Long::valueOf)
                .collect(Collectors.toList());
    }

    @Nullable
    public static List<Integer> metrikaCountersFromDbToList(String counters) {
        if (counters == null) {
            return null;
        }
        if (counters.isEmpty()) {
            return Collections.emptyList();
        }
        return StreamEx.split(counters, METRIKA_COUNTERS_DELIMITER)
                .map(Integer::valueOf)
                .toList();
    }

    @Nullable
    public static List<Long> metrikaCountersFromDbToListOfCounters(String counters) {
        return mapList(metrikaCountersFromDbToList(counters), Long::valueOf);
    }

    public static String metrikaCountersToDb(List<Long> counters) {
        if (counters == null) {
            return null;
        }
        return counters.stream()
                .map(Object::toString)
                .collect(Collectors.joining(METRIKA_COUNTERS_DELIMITER));
    }

    public static String metrikaCounterIdsToDb(List<Long> counters) {
        if (counters == null) {
            return null;
        }
        return counters.stream()
                .map(Object::toString)
                .collect(Collectors.joining(METRIKA_COUNTERS_DELIMITER));
    }

    /**
     * Преобразует список запрещенных площадок в json-массив
     *
     * @param pageIds список площадок на которых допустимо показывать материалы кампании
     * @return json-массив
     */
    @Nullable
    public static String pageIdsToDb(@Nullable List<Long> pageIds) {
        if (pageIds != null) {
            pageIds = pageIds.stream()
                    .distinct()
                    .collect(Collectors.toList());
        }

        return JsonUtils.toJsonCollectionEmptyToNull(pageIds);
    }

    /**
     * @param allowedPageIdsJson строка с JSON-массивом, содержащим список площадок
     *                           на которых допустимо показывать материалы кампании
     * @return список площадок
     */
    @Nullable
    public static List<Long> allowedPageIdsFromDb(@Nullable String allowedPageIdsJson) {
        return isNotEmpty(allowedPageIdsJson)
                ? JsonUtils.fromJson(allowedPageIdsJson, new TypeReference<List<Long>>() {
        })
                : Collections.emptyList();
    }

    /**
     * @param pageIdsJson строка с JSON-массивом, содержащим список площадок
     * Используется для списка разрешенных площадок: camp_options.allowed_page_ids,
     * А также для списка запрещенных площадок: camp_options.disallowed_page_ids
     *
     * @return список площадок, null если площадок нет
     */
    @Nullable
    public static List<Long> pageIdsFromDbNullWhenEmpty(@Nullable String pageIdsJson) {
        return isNotEmpty(pageIdsJson)
                ? JsonUtils.fromJson(pageIdsJson, new TypeReference<List<Long>>() {
        })
                : null;
    }

    /**
     * @param domains запрещенные домены через запятую
     * @return сет доменов
     */
    @Nullable
    public static Set<String> disabledDomainsFromDb(@Nullable String domains) {
        return domains == null || domains.isEmpty() ? null : Arrays.stream(domains.split(","))
                .collect(toSet());
    }

    /**
     * Преобразует список запрещенных доменов в строку через запятую
     *
     * @param domains сет доменов
     * @return запрещенные домены через запятую
     */
    @Nullable
    public static String disabledDomainsToDb(@Nullable Set<String> domains) {
        return domains == null ? null : domains.stream()
                .sorted()
                .collect(joining(DISABLED_DOMAINS_DELIMITER));
    }

    /**
     * @param domains запрещенные домены через запятую
     * @return лист доменов
     */
    @Nullable
    public static List<String> disabledDomainsFromDbToList(@Nullable String domains) {
        return domains == null || domains.isEmpty() ? null : List.of(domains.split(DISABLED_DOMAINS_DELIMITER));
    }

    /**
     * Преобразует список запрещенных доменов в строку через запятую
     *
     * @param domains лист доменов
     * @return запрещенные домены через запятую
     */
    @Nullable
    public static String disabledDomainsToDb(@Nullable List<String> domains) {
        return domains == null ? null : domains.stream()
                .sorted()
                .collect(joining(DISABLED_DOMAINS_DELIMITER));
    }

    /**
     * @param ips запрещенные ip адреса через запятую
     * @return лист ip адресов
     */
    @Nullable
    public static List<String> disabledIpsFromDbToList(@Nullable String ips) {
        return org.apache.commons.lang3.StringUtils.isNotEmpty(ips) ?
                List.of(ips.split(DISABLED_IPS_DELIMITER)) : null;
    }

    /**
     * Преобразует список запрещенных ip адресов в строку через запятую
     *
     * @param ips лист доменов
     * @return ip адреса через запятую
     */
    @Nullable
    public static String disabledIpsToDb(@Nullable List<String> ips) {
        return ips == null ? null : ips.stream()
                .sorted()
                .collect(joining(DISABLED_IPS_DELIMITER));
    }

    /**
     * Преобразует список запрещенных площадок в json-массив
     *
     * @param disabledSsp список площадок
     * @return json-массив
     */
    @Nullable
    public static String disabledSspToJson(@Nullable List<String> disabledSsp) {
        return JsonUtils.toJsonNullable(disabledSsp);
    }

    /**
     * @param disabledSspJson запрещенные площадки в виде json-массива
     * @return список площадок
     */
    @Nullable
    @SuppressWarnings("unchecked")
    public static List<String> disabledSspFromJson(@Nullable String disabledSspJson) {
        return org.apache.commons.lang3.StringUtils.isNotEmpty(disabledSspJson) ? JsonUtils
                .fromJson(disabledSspJson, List.class) : null;
    }

    /**
     * Преобразует список запрещенных видео-площадок в json-массив
     *
     * @param disabledVideoPlacements список площадок
     * @return json-массив
     */
    @Nullable
    public static String disabledVideoPlacementsToJson(@Nullable List<String> disabledVideoPlacements) {
        return JsonUtils.toJsonNullable(disabledVideoPlacements);
    }

    /**
     * @param disabledVideoPlacementsJson запрещенные площадки в виде json-массива
     * @return список площадок
     */
    @Nullable
    @SuppressWarnings("unchecked")
    public static List<String> disabledVideoPlacementsFromJson(@Nullable String disabledVideoPlacementsJson) {
        return org.apache.commons.lang3.StringUtils.isNotEmpty(disabledVideoPlacementsJson) ? JsonUtils
                .fromJson(disabledVideoPlacementsJson, List.class) : null;
    }

    @Nullable
    public static Set<Integer> geoFromDb(@Nullable String geo) {
        return geo == null ? null : Arrays.stream(geo.split(","))
                .map(Integer::valueOf)
                .collect(toSet());
    }

    @Nullable
    public static String geoToDb(@Nullable Set<Integer> geo) {
        return geo == null ? null : geo.stream()
                .map(Object::toString)
                .collect(joining(","));
    }

    /**
     * Конвертирует строку в {@link TimeInterval}
     *
     * @param smsTime интервал времени в формате {hh:mm:hh:mm}
     * @return {@link TimeInterval}
     */
    public static TimeInterval smsTimeFromDb(String smsTime) throws IllegalStateException,
            NumberFormatException {
        String[] parts = smsTime.split(":");
        checkState(parts.length == 4, "invalid time interval format");

        return new TimeInterval()
                .withStartHour(Integer.parseInt(parts[0]))
                .withStartMinute(Integer.parseInt(parts[1]))
                .withEndHour(Integer.parseInt(parts[2]))
                .withEndMinute(Integer.parseInt(parts[3]));
    }

    public static String smsTimeToDb(TimeInterval smsTime) {
        return String.format("%02d:%02d:%02d:%02d", smsTime.getStartHour(), smsTime.getStartMinute(),
                smsTime.getEndHour(), smsTime.getEndMinute());
    }

    public static CampOptionsBroadMatchFlag broadMatchFlagToDb(Boolean broadMatchFlag) {
        return (broadMatchFlag != null && broadMatchFlag)
                ? CampOptionsBroadMatchFlag.Yes
                : CampOptionsBroadMatchFlag.No;
    }

    public static Boolean broadMatchFlagFromDb(CampOptionsBroadMatchFlag broadMatchFlag) {
        return broadMatchFlag == CampOptionsBroadMatchFlag.Yes;
    }

    public static TimeTarget timeTargetFromDb(@Nullable String timeTargetString) {
        return timeTargetString == null ? defaultTimeTarget() : TimeTarget.parseRawString(timeTargetString);
    }

    public static String timeTargetToDb(@Nullable TimeTarget timeTarget) {
        // Убираем пресеты из таймтаргетинга или выставляем null, если включены круглосуточные показы
        boolean isDefault = defaultTimeTarget().equals(timeTarget);
        return timeTarget == null || isDefault ? null : timeTarget.withPreset(null).toRawFormat();
    }

    public static boolean isAutoPayFromDb(WalletCampaignsAutopayMode mode) {
        return WalletCampaignsAutopayMode.min_balance == mode;
    }

    public static WalletCampaignsAutopayMode isAutoPayToDb(boolean isAutoPay) {
        return isAutoPay ? WalletCampaignsAutopayMode.min_balance : WalletCampaignsAutopayMode.none;
    }

    public static String stringListToDbJsonFormat(List<String> list) {
        return list == null ? null : JsonUtils.toJson(list);
    }

    public static List<String> stringListFromDbJsonFormat(String json) {
        if (json == null) {
            return null;
        }
        return Arrays.asList(JsonUtils.fromJson(json, String[].class));
    }
}
