package ru.yandex.direct.core.entity.freelancer.model;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;

/**
 * ВАЖНО!!!
 * Эти коды нельзя переименовывать и/или удалять, т.к. они используются в том числе во внешних системах.
 * <p>
 * Удаление производим через выставление флага {@link #active} в {@code false}.
 * Такие услуги не попадают в {@link #allFreelancerSkills()} и отфильтровываются в
 * {@code ru.yandex.direct.core.entity.freelancer.service.FreelancerSkillOffersService#getFreelancerSkillsOffers(java.util.List)}
 */
@ParametersAreNonnullByDefault
public enum FreelancerSkill {
    SETTING_UP_CAMPAIGNS_FROM_SCRATCH(1, "Настройка рекламных кампаний с нуля", true),
    CAMPAIGN_CONDUCTING(2, "Ведение рекламных кампаний", true),
    CAMPAIGN_AUDIT(3, "Аудит и рекомендации по настройке", true),
    METRIKA_SETUP(4, "Настройка Яндекс.Метрики", false),
    ;

    private static final List<FreelancerSkill> ALL_CODES =
            StreamEx.of(FreelancerSkill.values())
                    .filter(FreelancerSkill::isActive)
                    .toListAndThen(Collections::unmodifiableList);

    /**
     * Соответсвие между идентификатором услуги и самой услугой
     */
    private static final ImmutableMap<Long, FreelancerSkill> SKILL_BY_ID =
            StreamEx.of(FreelancerSkill.values())
                    .mapToEntry(FreelancerSkill::getSkillId)
                    .invert()
                    .toMapAndThen(ImmutableMap::copyOf);

    private final long skillId;
    private final String description;
    private final boolean active;

    FreelancerSkill(long skillId, String description, boolean active) {
        this.skillId = skillId;
        this.description = description;
        this.active = active;
    }

    public Long getSkillId() {
        return skillId;
    }

    public String getSkillCode() {
        return this.name();
    }

    public String getDescription() {
        return this.description;
    }

    public boolean isActive() {
        return active;
    }

    /**
     * Все активные услуги.
     * Активность определяется параметром {@link #isActive()}
     */
    public static List<FreelancerSkill> allFreelancerSkills() {
        return ALL_CODES;
    }

    /**
     * Ищет активную услугу {@link FreelancerSkill} по {@code skillId}.
     */
    @Nonnull
    private static Optional<FreelancerSkill> findActiveById(Long skillId) {
        return Optional.ofNullable(SKILL_BY_ID.get(skillId))
                .filter(FreelancerSkill::isActive);
    }

    /**
     * Проверка, что услуга с таким {@code skillId} существует и активна.
     */
    public static boolean isActiveSkillId(Long skillId) {
        return findActiveById(skillId)
                .isPresent();
    }

    /**
     * Возвращает активную {@link FreelancerSkill} по заданному {@code skillId}
     * или бросает {@link IllegalArgumentException} если такой услуги нет
     */
    @Nonnull
    public static FreelancerSkill getById(Long skillId) {
        return findActiveById(skillId)
                .orElseThrow(() -> new IllegalArgumentException("No FreelancerSkill with id " + skillId));
    }
}
