package ru.yandex.direct.core.entity.promoextension.model

import com.google.common.base.Preconditions.checkState
import java.time.LocalDate
import java.time.Month

import ru.yandex.direct.core.entity.ClientLibraryObject
import ru.yandex.direct.core.entity.moderation.model.Moderationable
import ru.yandex.direct.core.entity.moderation.model.TransportStatus
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsPrefix
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsPrefix.from
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsPrefix.to
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.cashback
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.discount
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.free
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.gift
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.installment
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.profit
import ru.yandex.direct.model.Entity
import java.text.DecimalFormat
import java.text.DecimalFormatSymbols

//val соответствует клиентским полям, var - служебным, так как мы сами проставляем id и меняем статус
data class PromoExtension(
    var promoExtensionId: Long?,
    var clientId: ClientId,
    val type: PromoactionsType,
    val amount: Long?,
    val unit: PromoExtensionUnit?,
    val prefix: PromoactionsPrefix?,
    val href: String?,
    val description: String,
    val startDate: LocalDate?,
    val finishDate: LocalDate?,
    var statusModerate: PromoactionsStatusmoderate,
) : DataToExtractPromoExtensionCompoundDescription(
    type, amount, unit, prefix, description, finishDate
), Entity<Long>, ClientLibraryObject {
    override fun getId(): Long? {
        return promoExtensionId
    }

    override fun setId(id: Long?) {
        promoExtensionId = id
    }
}

enum class PromoExtensionUnit(
    val dbValue: String,
    val valueInDescription: String,
    val minAmountValue: Long,
    val maxAmountValue: Long,
) {
    RUB("rub", "\u00a0₽", 1L, 100_000_000L),
    PCT("pct", "%", 1L, 100L),
    ;

    companion object {
        private val ignored: Unit = checkState(values().all { it.dbValue.length == 3 }) //в БД unit это char(3)
        private val byDbValue: Map<String, PromoExtensionUnit> = values().associateBy { it.dbValue }

        fun fromDbValue(promoExtensionUnit: String) = byDbValue[promoExtensionUnit]
            ?: throw IllegalArgumentException("Invalid core promoextension unit: $promoExtensionUnit")
    }
}

data class PromoExtensionWithModerationInfo(
    private var clientId: Long,
    private var uid: Long,
    private var version: Long,
    private var transportStatus: TransportStatus,
    val promoExtensionId: Long,
    val description: String,
    val href: String?,
    val type: PromoactionsType,
    val amount: Long?,
    val unit: PromoExtensionUnit?,
    val prefix: PromoactionsPrefix?,
    val finishDate: LocalDate?,
    val clientFlags: Set<String>,
) : Moderationable, DataToExtractPromoExtensionCompoundDescription(
    type, amount, unit, prefix, description, finishDate
) {
    override fun getClientId() = clientId
    override fun getUid() = uid
    override fun getVersion() = version
    override fun getTransportStatus() = transportStatus

    override fun setClientId(clientId: Long) {
        this.clientId = clientId
    }

    override fun withClientId(clientId: Long): PromoExtensionWithModerationInfo {
        setClientId(clientId)
        return this
    }

    override fun setTransportStatus(transportStatus: TransportStatus) {
        this.transportStatus = transportStatus
    }

    override fun withTransportStatus(transportStatus: TransportStatus): PromoExtensionWithModerationInfo {
        setTransportStatus(transportStatus)
        return this
    }

    override fun setUid(uid: Long) {
        this.uid = uid
    }

    override fun withUid(uid: Long): PromoExtensionWithModerationInfo {
        setUid(uid)
        return this
    }

    override fun setVersion(version: Long) {
        this.version = version
    }

    override fun withVersion(version: Long): PromoExtensionWithModerationInfo {
        setVersion(version)
        return this
    }
}

open class DataToExtractPromoExtensionCompoundDescription(
        private val type: PromoactionsType,
        private val amount: Long?,
        private val unit: PromoExtensionUnit?,
        private val prefix: PromoactionsPrefix?,
        private val description: String,
        private val finishDate: LocalDate?,
) {
    companion object {
        private const val DIGIT_NUMBER_TO_USE_DECIMAL_FORMAT = 5

        private val DECIMAL_FORMAT: DecimalFormat

        init {
            val formatSymbols = DecimalFormatSymbols()
            formatSymbols.groupingSeparator = '\u00a0'
            DECIMAL_FORMAT = DecimalFormat("#,###", formatSymbols)
        }

        private fun monthToDesc(month: Month) =
            when (month) {
                Month.JANUARY -> "января"
                Month.FEBRUARY -> "февраля"
                Month.MARCH -> "марта"
                Month.APRIL -> "апреля"
                Month.MAY -> "мая"
                Month.JUNE -> "июня"
                Month.JULY -> "июля"
                Month.AUGUST -> "августа"
                Month.SEPTEMBER -> "сентября"
                Month.OCTOBER -> "октября"
                Month.NOVEMBER -> "ноября"
                Month.DECEMBER -> "декабря"
            }

        private fun typeToDesc(type: PromoactionsType) =
            when (type) {
                cashback -> "кешбэк"
                discount -> "скидка"
                free -> "бесплатно"
                gift -> "в подарок"
                installment -> "рассрочка 0%"
                profit -> "выгода"
            }

        private fun prefixToDesc(prefix: PromoactionsPrefix) =
            when (prefix) {
                from -> "от"
                to -> "до"
            }

        private fun amountToDesc(amount: Long) =
            if (amount.toString().length >= DIGIT_NUMBER_TO_USE_DECIMAL_FORMAT) {
                DECIMAL_FORMAT.format(amount)
            } else {
                amount.toString()
            }
    }

    val compoundDescriptionWithoutDate: String get() = buildCompoundDescriptionWithoutDate().toString()

    val compoundDescription: String get() = buildCompoundDescriptionWithDate().toString()

    private fun buildCompoundDescriptionWithoutDate(): StringBuilder {
        val result = StringBuilder()
        if (!(type in setOf(free, gift))) {
            result.append(typeToDesc(type).capitalize())
        }
        if (type in setOf(cashback, discount, profit)) {
            result.append(prefix?.let { " " + prefixToDesc(it) } ?: "")
            result.append(amount?.let { " " + amountToDesc(it) } ?: "")
            result.append(unit?.valueInDescription.orEmpty())
        }
        if (type in setOf(free, gift)) {
            result.append(description.capitalize())
        } else {
            result.append(" ")
            result.append(description.decapitalize())
        }
        if (type in setOf(free, gift)) {
            result.append(" ")
            result.append(typeToDesc(type))
            result.append("!")
        }
        return result
    }

    private fun buildCompoundDescriptionWithDate(): StringBuilder {
        val result = buildCompoundDescriptionWithoutDate()
        if (finishDate != null) {
            result.append(if (result.trim().matches(Regex(".*[.?!]$"))) {
                " До "
            } else {
                " до "
            })
            result.append(finishDate.dayOfMonth)
            result.append(" ")
            result.append(monthToDesc(finishDate.month))
        }
        return result
    }
}
