package ru.yandex.direct.core.entity.retargeting.model;

import java.util.Objects;
import java.util.Set;

import static ru.yandex.direct.core.entity.retargeting.Constants.AUDIENCE_TIME_VALUE;
import static ru.yandex.direct.core.entity.retargeting.Constants.CDP_SEGMENT_TIME_VALUE;
import static ru.yandex.direct.core.entity.retargeting.Constants.CRYPTA_GOAL_TIME_VALUE;
import static ru.yandex.direct.core.entity.retargeting.Constants.SEGMENT_TIME_VALUE;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class Goal extends GoalBase {

    //Верхние границы типов по идентификатору не включительно
    //Например интервал типа SEGMENT: с METRIKA_GOAL_UPPER_BOUND по METRIKA_SEGMENT_UPPER_BOUND - 1
    //Актуальные данные о диапазонах goal_id в Метрике: https://wiki.yandex-team.ru/JurijjGalickijj/raznoe/goalid/
    public static final long METRIKA_GOAL_UPPER_BOUND = 1_000_000_000L;
    public static final long METRIKA_SEGMENT_UPPER_BOUND = 1_500_000_000L;
    public static final long LAL_SEGMENT_UPPER_BOUND = 1_900_000_000L;
    public static final long MOBILE_GOAL_UPPER_BOUND = 2_000_000_000L;
    public static final long METRIKA_AUDIENCE_LOWER_BOUND = 2_000_000_000L;
    public static final long METRIKA_AUDIENCE_UPPER_BOUND = 2_499_000_000L;
    public static final long CRYPTA_SOCIAL_DEMO_UPPER_BOUND = 2_499_000_100L;
    public static final long CRYPTA_FAMILY_UPPER_BOUND = 2_499_000_200L;
    public static final long CRYPTA_BEHAVIORS_UPPER_BOUND = 2_499_001_100L;
    public static final long CRYPTA_INTERESTS_UPPER_BOUND = 2_499_980_000L;
    //Для CUSTOM_AUDIENCE записываем все в свой диапазон, т.к. остальное записывается руками
    public static final long CRYPTA_INTERESTS_CA_LOWER_BOUND = CRYPTA_INTERESTS_UPPER_BOUND - 300_000L;
    public static final long CRYPTA_INTERNAL_UPPER_BOUND = 2_499_990_000L;
    public static final long MUSIC_AUDIO_GENRES_UPPER_BOUND = 2_500_000_000L;
    public static final long AB_SEGMENT_UPPER_BOUND = 2_600_000_000L;
    public static final long CDP_SEGMENT_UPPER_BOUND = 3_000_000_000L;
    public static final long METRIKA_ECOMMERCE_UPPER_BOUND = 3_900_000_000L;
    public static final long METRIKA_COUNTER_LOWER_BOUND = 4_000_000_000L;
    public static final long METRIKA_COUNTER_UPPER_BOUND = 4_100_000_000L;
    public static final long BRANDSAFETY_LOWER_BOUND = 4_294_967_296L; // 2^32
    public static final long BRANDSAFETY_UPPER_BOUND = BRANDSAFETY_LOWER_BOUND + 1_000L;
    public static final long CONTENT_CATEGORY_UPPER_BOUND = BRANDSAFETY_LOWER_BOUND + 3_000L;
    public static final long CONTENT_GENRE_UPPER_BOUND = BRANDSAFETY_LOWER_BOUND + 5_000L;
    public static final long HOST_LOWER_BOUND = 19_000_000_000L;
    public static final long HOST_UPPER_BOUND = 19_900_000_000L;

    // id ecommerce цели получается как 3_000_000_000L + id счетчика цели
    public static final long METRIKA_ECOMMERCE_BASE = 3_000_000_000L;

    // Назначим фиктивному "Жанры кино" отрицательный id, чтобы не пересекался с реальными id из базы.
    public static final Long CINEMA_GENRES_GOAL_ID = -42L;

    public static final Set<Long> MOBILE_GOAL_IDS = Set.of(
            3L, 4L, 5L, 6L, 7L,
            38402972L, 38403008L, 38403053L, 38403071L, 38403080L, 38403095L, 38403104L,
            38403131L, 38403173L, 38403191L, 38403197L, 38403206L, 38403215L, 38403230L,
            38403338L, 38403494L, 38403530L, 38403545L, 38403581L
    );

    public static final Set<GoalType> TYPES_WITH_AUDIENCE_TIME = Set.of(
            GoalType.AUDIENCE, GoalType.LAL_SEGMENT, GoalType.HOST
    );

    @Override
    public void setId(Long id) {
        super.setId(id);
        super.setType(computeType(id));
        if (id != null) {
            super.setIsMobileGoal(MOBILE_GOAL_IDS.contains(id));
        }
        if (getType() != null && TYPES_WITH_AUDIENCE_TIME.contains(getType())) {
            super.setTime(AUDIENCE_TIME_VALUE);
        } else if (GoalType.SEGMENT.equals(getType())) {
            super.setTime(SEGMENT_TIME_VALUE);
        } else if (GoalType.CDP_SEGMENT.equals(getType())) {
            super.setTime(CDP_SEGMENT_TIME_VALUE);
        } else if (getType() != null && getType().isCrypta()) {
            super.setTime(CRYPTA_GOAL_TIME_VALUE);
        }
    }

    /**
     * Получить тип цели по идентификатору
     * <p>
     * Актуальная информация о мапинге goal_id в тип цели
     * <a href="https://wiki.yandex-team.ru/JurijjGalickijj/raznoe/goalid/">на вики</a>.
     *
     * @return {@link GoalType} тип цели
     */
    public static GoalType computeType(Long id) {
        if (id == null) {
            return null;
        } else if (id.equals(CINEMA_GENRES_GOAL_ID)) {
            // Этот id отсутствует в таблицах crypta_goals или metrika_goals
            // Но присутствует в ретагретингах в других таблицах
            return GoalType.CONTENT_GENRE;
        } else if (id < 0) {
            return null;
        } else if (id < METRIKA_GOAL_UPPER_BOUND) {
            return GoalType.GOAL;
        } else if (id < METRIKA_SEGMENT_UPPER_BOUND) {
            return GoalType.SEGMENT;
        } else if (id < LAL_SEGMENT_UPPER_BOUND) {
            return GoalType.LAL_SEGMENT;
        } else if (id < MOBILE_GOAL_UPPER_BOUND) {
            return GoalType.MOBILE;
        } else if (id < METRIKA_AUDIENCE_UPPER_BOUND) {
            return GoalType.AUDIENCE;
        } else if (id < CRYPTA_SOCIAL_DEMO_UPPER_BOUND) {
            return GoalType.SOCIAL_DEMO;
        } else if (id < CRYPTA_FAMILY_UPPER_BOUND) {
            return GoalType.FAMILY;
        } else if (id < CRYPTA_BEHAVIORS_UPPER_BOUND) {
            return GoalType.BEHAVIORS;
        } else if (id < CRYPTA_INTERESTS_UPPER_BOUND) {
            return GoalType.INTERESTS;
        } else if (id < CRYPTA_INTERNAL_UPPER_BOUND) {
            return GoalType.INTERNAL;
        } else if (id < MUSIC_AUDIO_GENRES_UPPER_BOUND) {
            return GoalType.AUDIO_GENRES;
        } else if (id < AB_SEGMENT_UPPER_BOUND) {
            return GoalType.AB_SEGMENT;
        } else if (id < CDP_SEGMENT_UPPER_BOUND) {
            return GoalType.CDP_SEGMENT;
        } else if (id < METRIKA_ECOMMERCE_UPPER_BOUND) {
            return GoalType.ECOMMERCE;
        } else if (id < BRANDSAFETY_LOWER_BOUND) {
            return GoalType.GOAL;
        } else if (id < BRANDSAFETY_UPPER_BOUND) {
            return GoalType.BRANDSAFETY;
        } else if (id < CONTENT_CATEGORY_UPPER_BOUND) {
            return GoalType.CONTENT_CATEGORY;
        } else if (id < CONTENT_GENRE_UPPER_BOUND) {
            return GoalType.CONTENT_GENRE;
        } else if (id >= HOST_LOWER_BOUND && id < HOST_UPPER_BOUND) {
            return GoalType.HOST;
        } else {
            return GoalType.GOAL;
        }
    }

    @Override
    public void setType(GoalType type) {
        //поле type зависит от id, поэтому определяем его, только если id == null
        if (this.getId() == null) {
            super.setType(type);
        }
    }

    @Override
    public void setTime(Integer time) {
        //поле time для аудиторий/сегментов Метрики/cdp сегментов и LAL-сегментов должно быть всегда 540,
        //а для целей крипты - 0
        if (getType() != null && !getType().isCrypta() && !GoalType.AUDIENCE.equals(getType())
                && !GoalType.SEGMENT.equals(getType()) && !GoalType.CDP_SEGMENT.equals(getType())) {
            super.setTime(time);
        }
    }

    /**
     * Является ли цель родительской составной целью
     */
    public boolean isStepGoalParent() {
        return Objects.equals(super.getMetrikaCounterGoalType(), MetrikaCounterGoalType.STEP)
                && nvl(super.getParentId(), 0L) == 0;
    }

    /**
     * Является ли цель дочерней составной целью
     */
    public boolean isStepGoalChild() {
        return nvl(super.getParentId(), 0L) != 0;
    }
}
