package ru.yandex.direct.core.entity.strategy;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.model.StrategyType;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Map.entry;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName.autobudget;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName.autobudget_avg_click;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName.autobudget_avg_cpa;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName.autobudget_avg_cpa_per_camp;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName.autobudget_avg_cpi;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName.autobudget_week_bundle;

public class StrategyExtractorHelper {

    private static final Logger logger = LoggerFactory.getLogger(StrategyExtractorHelper.class);

    private static final Map<CampaignsStrategyName, List<StrategyType>> STRATEGY_TYPES_BY_STRATEGY_NAME =
            Map.ofEntries(
                    entry(CampaignsStrategyName.default_, List.of(StrategyType.DEFAULT)),
                    entry(CampaignsStrategyName.no_premium, List.of(StrategyType.NO_PREMIUM)),
                    entry(CampaignsStrategyName.autobudget, List.of(StrategyType.WEEK_BUDGET,
                            StrategyType.OPTIMIZE_CLICKS, StrategyType.OPTIMIZE_CONVERSIONS,
                            StrategyType.OPTIMIZE_INSTALLS)),
                    entry(CampaignsStrategyName.autobudget_week_bundle, List.of(StrategyType.WEEK_BUNDLE,
                            StrategyType.OPTIMIZE_CLICKS)),
                    entry(CampaignsStrategyName.autobudget_avg_click, List.of(StrategyType.AVG_CLICK,
                            StrategyType.OPTIMIZE_CLICKS)),
                    entry(CampaignsStrategyName.autobudget_avg_cpa, List.of(StrategyType.AVG_CPA,
                            StrategyType.OPTIMIZE_CONVERSIONS)),
                    entry(CampaignsStrategyName.autobudget_avg_cpa_per_camp, List.of(StrategyType.AVG_CPA_PER_CAMP,
                            StrategyType.OPTIMIZE_CONVERSIONS)),
                    entry(CampaignsStrategyName.autobudget_avg_cpa_per_filter,
                            List.of(StrategyType.AVG_CPA_PER_FILTER)),
                    entry(CampaignsStrategyName.autobudget_avg_cpc_per_camp, List.of(StrategyType.AVG_CPC_PER_CAMP)),
                    entry(CampaignsStrategyName.autobudget_avg_cpc_per_filter,
                            List.of(StrategyType.AVG_CPC_PER_FILTER)),
                    entry(CampaignsStrategyName.autobudget_avg_cpi, List.of(StrategyType.AVG_CPI,
                            StrategyType.OPTIMIZE_INSTALLS)),
                    entry(CampaignsStrategyName.autobudget_roi, List.of(StrategyType.ROI)),
                    entry(CampaignsStrategyName.autobudget_crr, List.of(StrategyType.CRR)),
                    entry(CampaignsStrategyName.cpm_default, List.of(StrategyType.CPM_DEFAULT)),
                    entry(CampaignsStrategyName.autobudget_max_reach, List.of(StrategyType.CPM_MAX_REACH)),
                    entry(CampaignsStrategyName.autobudget_max_reach_custom_period,
                            List.of(StrategyType.CPM_MAX_REACH_CUSTOM_PERIOD)),
                    entry(CampaignsStrategyName.autobudget_max_impressions, List.of(StrategyType.CPM_MAX_IMPRESSIONS)),
                    entry(CampaignsStrategyName.autobudget_max_impressions_custom_period,
                            List.of(StrategyType.CPM_MAX_IMPRESSIONS_CUSTOM_PERIOD)),
                    entry(CampaignsStrategyName.autobudget_avg_cpv, List.of(StrategyType.AVG_CPV)),
                    entry(CampaignsStrategyName.autobudget_avg_cpv_custom_period,
                            List.of(StrategyType.AVG_CPV_CUSTOM_PERIOD)),
                    entry(CampaignsStrategyName.period_fix_bid, List.of(StrategyType.PERIOD_FIX_BID))
            );

    public static StrategyType toStrategyType(
            CampaignsStrategyName strategyName,
            StrategyData strategyData,
            CampaignType campaignType
    ) {
        checkArgument(STRATEGY_TYPES_BY_STRATEGY_NAME.containsKey(strategyName),
                "Got campaignStrategyGroup with unsupported strategy " + strategyName);
        List<StrategyType> possibleTypes = STRATEGY_TYPES_BY_STRATEGY_NAME.get(strategyName);
        if (possibleTypes.size() == 1) {
            return possibleTypes.get(0);
        }
        return detectFinalStrategyType(
                possibleTypes,
                strategyName,
                strategyData,
                campaignType
        );
    }

    @Nullable
    public static StrategyType toStrategyTypeSafely(
            StrategyName strategyName,
            StrategyData strategyData,
            CampaignType campaignType
    ) {
        var campStrategyName = StrategyName.toSource(strategyName);
        if (strategyName == null || !STRATEGY_TYPES_BY_STRATEGY_NAME.containsKey(campStrategyName)) {
            logger.warn("Got campaignStrategyGroup with unsupported strategy " + strategyName);
            return null;
        }
        return toStrategyType(campStrategyName, strategyData, campaignType);
    }

    private static StrategyType detectFinalStrategyType(
            List<StrategyType> possibleTypes,
            CampaignsStrategyName strategyName,
            StrategyData strategyData,
            CampaignType campaignType
    ) {
        if (possibleTypes.contains(StrategyType.OPTIMIZE_CLICKS)) {
            if (isOptimizeClicksStrategy(strategyName, strategyData)) {
                return StrategyType.OPTIMIZE_CLICKS;
            }
        }
        if (possibleTypes.contains(StrategyType.OPTIMIZE_CONVERSIONS)) {
            if (isOptimizeConversionsStrategy(campaignType, strategyName, strategyData)) {
                return StrategyType.OPTIMIZE_CONVERSIONS;
            }
        }
        if (possibleTypes.contains(StrategyType.OPTIMIZE_INSTALLS)) {
            if (isOptimizeInstallsStrategy(campaignType, strategyName, strategyData)) {
                return StrategyType.OPTIMIZE_INSTALLS;
            }
        }
        if (possibleTypes.contains(StrategyType.AVG_CPA)) {
            return StrategyType.AVG_CPA;
        }
        if (possibleTypes.contains(StrategyType.WEEK_BUDGET)) {
            return StrategyType.WEEK_BUDGET;
        }
        if (possibleTypes.contains(StrategyType.WEEK_BUNDLE)) {
            return StrategyType.WEEK_BUNDLE;
        }
        if (possibleTypes.contains(StrategyType.AVG_CPI)) {
            return StrategyType.AVG_CPI;
        }
        if (possibleTypes.contains(StrategyType.AVG_CLICK)) {
            return StrategyType.AVG_CLICK;
        }
        throw new IllegalStateException("Cannot define strategy type");
    }

    private static boolean isOptimizeClicksStrategy(CampaignsStrategyName strategyName,
                                                    @Nullable StrategyData strategyData) {
        return autobudget_week_bundle == strategyName || autobudget_avg_click == strategyName ||
                (autobudget == strategyName
                        && null != strategyData
                        && null == strategyData.getGoalId());
    }

    private static boolean isOptimizeConversionsStrategy(CampaignType campaignType,
                                                         CampaignsStrategyName strategyName,
                                                         @Nullable StrategyData strategyData) {
        return autobudget_avg_cpa == strategyName ||
                autobudget_avg_cpa_per_camp == strategyName ||
                (CampaignType.MOBILE_CONTENT != campaignType
                        && autobudget == strategyName
                        && null != strategyData
                        && null != strategyData.getGoalId());
    }

    private static boolean isOptimizeInstallsStrategy(CampaignType campaignType,
                                                      CampaignsStrategyName strategyName,
                                                      @Nullable StrategyData strategyData) {
        return autobudget_avg_cpi == strategyName ||
                (CampaignType.MOBILE_CONTENT == campaignType
                        && autobudget == strategyName
                        && null != strategyData
                        && null != strategyData.getGoalId());
    }
}
