package ru.yandex.direct.dbqueue.repository;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.dbqueue.DbQueueJobType;
import ru.yandex.direct.dbqueue.LimitOffset;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.model.DbQueueJobStatus;
import ru.yandex.direct.dbschema.ppc.enums.DbqueueJobArchiveStatus;
import ru.yandex.direct.dbschema.ppc.enums.DbqueueJobsStatus;
import ru.yandex.direct.dbschema.ppc.tables.records.DbqueueJobArchiveRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.DbqueueJobsRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.utils.SystemUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Lists.partition;
import static java.time.LocalDateTime.now;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toUnmodifiableList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.DBQUEUE_JOBS;
import static ru.yandex.direct.dbschema.ppc.Tables.DBQUEUE_JOB_ARCHIVE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class DbQueueRepository {
    public static final Duration DEFAULT_GRAB_DURATION = Duration.ofMinutes(15);

    private static final int MAX_GRABBED_BY_LENGTH = DBQUEUE_JOBS.GRABBED_BY.getDataType().length();

    // в базе там mediumblob
    private static final int MAX_PACKED_ARGS_LENGTH = (1 << 24) - 1;
    private static final int MAX_PACKED_RESULT_LENGTH = (1 << 24) - 1;

    private static final int MAX_ATTEMPTS_TO_MOVE_TO_ARCHIVE = 3;
    private static final int DELETION_CHUNK_SIZE = 100;

    private static final JooqMapper<DbQueueJob> JOBS_MAPPER =
            JooqMapperBuilder.<DbQueueJob>builder()
                    .map(property(DbQueueJob.ID, DBQUEUE_JOBS.JOB_ID))
                    .map(clientIdProperty(DbQueueJob.CLIENT_ID, DBQUEUE_JOBS.CLIENT_ID))
                    .map(property(DbQueueJob.UID, DBQUEUE_JOBS.UID))
                    .map(property(DbQueueJob.JOB_TYPE_ID, DBQUEUE_JOBS.JOB_TYPE_ID))
                    .map(property(DbQueueJob.NAMESPACE, DBQUEUE_JOBS.NAMESPACE))
                    .map(convertibleProperty(DbQueueJob.STATUS, DBQUEUE_JOBS.STATUS,
                            DbQueueJobStatus::fromJobsTableSource,
                            DbQueueJobStatus::toJobsTableSource))
                    .map(property(DbQueueJob.PRIORITY, DBQUEUE_JOBS.PRIORITY))
                    .map(property(DbQueueJob.CREATION_TIME, DBQUEUE_JOBS.CREATE_TIME))
                    .map(property(DbQueueJob.EXPIRATION_TIME, DBQUEUE_JOBS.EXPIRATION_TIME))
                    .map(property(DbQueueJob.GRABBED_BY, DBQUEUE_JOBS.GRABBED_BY))
                    .map(property(DbQueueJob.GRABBED_UNTIL, DBQUEUE_JOBS.GRABBED_UNTIL))
                    .map(property(DbQueueJob.GRABBED_AT, DBQUEUE_JOBS.GRABBED_AT))
                    .map(property(DbQueueJob.TRY_COUNT, DBQUEUE_JOBS.TRYCOUNT))
                    .map(property(DbQueueJob.RUN_AFTER, DBQUEUE_JOBS.RUN_AFTER))
                    .build();

    private static final JooqMapper<DbQueueJob> JOB_ARCHIVE_MAPPER =
            JooqMapperBuilder.<DbQueueJob>builder()
                    .map(property(DbQueueJob.ID, DBQUEUE_JOB_ARCHIVE.JOB_ID))
                    .map(clientIdProperty(DbQueueJob.CLIENT_ID, DBQUEUE_JOB_ARCHIVE.CLIENT_ID))
                    .map(property(DbQueueJob.UID, DBQUEUE_JOB_ARCHIVE.UID))
                    .map(property(DbQueueJob.JOB_TYPE_ID, DBQUEUE_JOB_ARCHIVE.JOB_TYPE_ID))
                    .map(property(DbQueueJob.NAMESPACE, DBQUEUE_JOB_ARCHIVE.NAMESPACE))
                    .map(convertibleProperty(DbQueueJob.STATUS, DBQUEUE_JOB_ARCHIVE.STATUS,
                            DbQueueJobStatus::fromJobArchiveTableSource,
                            DbQueueJobStatus::toJobArchiveTableSource))
                    .map(property(DbQueueJob.PRIORITY, DBQUEUE_JOB_ARCHIVE.PRIORITY))
                    .map(property(DbQueueJob.CREATION_TIME, DBQUEUE_JOB_ARCHIVE.CREATE_TIME))
                    .map(property(DbQueueJob.EXPIRATION_TIME, DBQUEUE_JOB_ARCHIVE.EXPIRATION_TIME))
                    .map(property(DbQueueJob.GRABBED_BY, DBQUEUE_JOB_ARCHIVE.GRABBED_BY))
                    .map(property(DbQueueJob.GRABBED_UNTIL, DBQUEUE_JOB_ARCHIVE.GRABBED_UNTIL))
                    .map(property(DbQueueJob.GRABBED_AT, DBQUEUE_JOB_ARCHIVE.GRABBED_AT))
                    .map(property(DbQueueJob.TRY_COUNT, DBQUEUE_JOB_ARCHIVE.TRYCOUNT))
                    .map(property(DbQueueJob.RUN_AFTER, DBQUEUE_JOB_ARCHIVE.RUN_AFTER))
                    .build();


    private static final List<Field<?>> JOBS_FIELDS_TO_READ;
    private static final List<Field<?>> JOB_ARCHIVE_FIELDS_TO_READ;
    private static final List<Field<?>> JOBS_FIELDS_TO_COPY_TO_ARCHIVE;
    private static final List<Field<?>> JOB_ARCHIVE_FIELDS_TO_FILL_WHEN_MARKING_DONE;
    private static final Logger LOGGER = LoggerFactory.getLogger(DbQueueRepository.class);

    static {
        List<Field<?>> jobsFieldsToRead = new ArrayList<>(JOBS_MAPPER.getFieldsToRead());
        jobsFieldsToRead.add(DBQUEUE_JOBS.ARGS);
        JOBS_FIELDS_TO_READ = Collections.unmodifiableList(jobsFieldsToRead);

        List<Field<?>> jobArchiveFieldsToRead = new ArrayList<>(JOB_ARCHIVE_MAPPER.getFieldsToRead());
        jobArchiveFieldsToRead.addAll(asList(DBQUEUE_JOB_ARCHIVE.ARGS, DBQUEUE_JOB_ARCHIVE.RESULT));
        JOB_ARCHIVE_FIELDS_TO_READ = Collections.unmodifiableList(jobArchiveFieldsToRead);

        JOBS_FIELDS_TO_COPY_TO_ARCHIVE = JOBS_FIELDS_TO_READ.stream()
                .filter(field -> !field.equals(DBQUEUE_JOBS.STATUS))
                .map(field -> field.as(field.getName()))
                .collect(toUnmodifiableList());

        List<Field<?>> jobArchiveFieldsToFillWhenMarkingDone = new ArrayList<>(JOBS_FIELDS_TO_COPY_TO_ARCHIVE);
        jobArchiveFieldsToFillWhenMarkingDone.addAll(asList(DSL.field(DBQUEUE_JOB_ARCHIVE.STATUS.getName()),
                DSL.field(DBQUEUE_JOB_ARCHIVE.RESULT.getName())));
        JOB_ARCHIVE_FIELDS_TO_FILL_WHEN_MARKING_DONE =
                Collections.unmodifiableList(jobArchiveFieldsToFillWhenMarkingDone);

    }

    private final DslContextProvider dslContextProvider;
    private final DbQueueTypeMap typeMap;
    private final ShardHelper shardHelper;

    public DbQueueRepository(DslContextProvider dslContextProvider,
                             ShardHelper shardHelper, DbQueueTypeMap typeMap) {
        this.dslContextProvider = dslContextProvider;
        this.typeMap = typeMap;
        this.shardHelper = shardHelper;
    }

    private static <A, R> DbQueueJob<A, R> modelFromJobsTable(DbQueueJobType<A, R> jobType, DbqueueJobsRecord record) {
        DbQueueJob<A, R> job = new DbQueueJob<>();

        JOBS_MAPPER.fromDb(record, job);

        job.setArgs(uncompress(record.get(DBQUEUE_JOBS.ARGS), jobType.getArgsClass()));

        return job;
    }

    private static <A, R> DbQueueJob<A, R> modelFromJobArchiveTable(DbQueueJobType<A, R> jobType,
                                                                    DbqueueJobArchiveRecord record) {
        DbQueueJob<A, R> job = new DbQueueJob<>();

        JOB_ARCHIVE_MAPPER.fromDb(record, job);

        job.setArgs(uncompress(record.get(DBQUEUE_JOB_ARCHIVE.ARGS), jobType.getArgsClass()));
        job.setResult(uncompress(record.get(DBQUEUE_JOB_ARCHIVE.RESULT), jobType.getResultClass()));

        return job;
    }

    @Nullable
    public <A, R> DbQueueJob<A, R> grabSingleJob(int shard, DbQueueJobType<A, R> jobType) {
        return grabSingleJob(shard, jobType, DEFAULT_GRAB_DURATION);
    }

    /**
     * Захватить задачу из очереди: записать в базу, что этот поток ей занимается, и увеличить в ней счётчик попыток
     * (tryCount) на 1.
     * <p>
     * Пространства имён (namespace, используются на бетах, чтобы задачей гарантированно занялся воркер с той же беты)
     * не поддерживаются. Захватываются только задачи, у которых namespace = NULL.
     *
     * @return захваченная задача или null, если таких не нашлось
     */
    @Nullable
    public <A, R> DbQueueJob<A, R> grabSingleJob(int shard, DbQueueJobType<A, R> jobType, Duration grabDuration) {

        Long jobTypeId = getJobTypeIdByJobType(jobType);

        String globalId = SystemUtils.hostname() + ":" + SystemUtils.getPid() + ":" + Thread.currentThread().getId();
        checkState(globalId.length() <= MAX_GRABBED_BY_LENGTH,
                "global id too long, max %d characters: %s", MAX_GRABBED_BY_LENGTH, globalId);

        int updatedRows = markJobsGrabbed(shard, grabDuration, jobTypeId, globalId, 1);
        if (updatedRows == 0) {
            return null;
        }

        DbqueueJobsRecord record = dslContextProvider.ppc(shard).select(JOBS_FIELDS_TO_READ)
                .from(DBQUEUE_JOBS)
                .where(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId))
                .and(DBQUEUE_JOBS.GRABBED_BY.eq(globalId))
                .limit(updatedRows)
                .fetchOne()
                .into(DBQUEUE_JOBS);

        DbQueueJob<A, R> job = modelFromJobsTable(jobType, record);
        LOGGER.debug("grabbed job: {}", job);

        return job;
    }

    public <A, R> List<DbQueueJob<A, R>> grabBunchOfJobs(
            int shard, DbQueueJobType<A, R> jobType, Duration grabDuration, int limit) {
        Long jobTypeId = getJobTypeIdByJobType(jobType);

        String globalId = SystemUtils.hostname() + ":" + SystemUtils.getPid() + ":" + Thread.currentThread().getId();
        checkState(globalId.length() <= MAX_GRABBED_BY_LENGTH,
                "global id too long, max %d characters: %s", MAX_GRABBED_BY_LENGTH, globalId);

        int updatedRows = markJobsGrabbed(shard, grabDuration, jobTypeId, globalId, limit);
        if (updatedRows == 0) {
            return emptyList();
        }

        var jobs = dslContextProvider.ppc(shard)
                .select(JOBS_FIELDS_TO_READ)
                .from(DBQUEUE_JOBS)
                .where(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId))
                .and(DBQUEUE_JOBS.GRABBED_BY.eq(globalId))
                .limit(updatedRows)
                .fetch()
                .into(DBQUEUE_JOBS)
                .map(record -> modelFromJobsTable(jobType, record));

        LOGGER.debug("grabbed {} jobs", jobs.size());

        return jobs;
    }

    @Nullable
    public <A, R> DbQueueJob<A, R> findJobById(int shard, DbQueueJobType<A, R> jobType, long id) {

        Long jobTypeId = getJobTypeIdByJobType(jobType);

        DbqueueJobsRecord jobsRecord = dslContextProvider.ppc(shard)
                .select(JOBS_FIELDS_TO_READ)
                .from(DBQUEUE_JOBS)
                .where(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId))
                .and(DBQUEUE_JOBS.JOB_ID.eq(id))
                .fetchOneInto(DBQUEUE_JOBS);

        if (jobsRecord != null) {
            return modelFromJobsTable(jobType, jobsRecord);
        }

        DbqueueJobArchiveRecord jobArchiveRecord = dslContextProvider.ppc(shard).select(JOB_ARCHIVE_FIELDS_TO_READ)
                .from(DBQUEUE_JOB_ARCHIVE)
                .where(DBQUEUE_JOB_ARCHIVE.JOB_TYPE_ID.eq(jobTypeId))
                .and(DBQUEUE_JOB_ARCHIVE.JOB_ID.eq(id))
                .fetchOneInto(DBQUEUE_JOB_ARCHIVE);

        if (jobArchiveRecord != null) {
            return modelFromJobArchiveTable(jobType, jobArchiveRecord);
        }

        return null;
    }

    /**
     * Найти все архивные джобы по idшникам
     *
     * @return все найденные в шарде джобы
     */
    public <A, R> List<DbQueueJob<A, R>> findArchivedJobsByIds(int shard, DbQueueJobType<A, R> jobType, Set<Long> ids) {
        Long jobTypeId = getJobTypeIdByJobType(jobType);

        var jobArchiveRecords = dslContextProvider.ppc(shard).select(JOB_ARCHIVE_FIELDS_TO_READ)
                .from(DBQUEUE_JOB_ARCHIVE)
                .where(DBQUEUE_JOB_ARCHIVE.JOB_TYPE_ID.eq(jobTypeId))
                .and(DBQUEUE_JOB_ARCHIVE.JOB_ID.in(ids))
                .fetchInto(DBQUEUE_JOB_ARCHIVE);

        return StreamEx.of(jobArchiveRecords)
                .map(j -> modelFromJobArchiveTable(jobType, j))
                .collect(Collectors.toList());
    }

    /**
     * Найти все архивные джобы со статусом failed
     *
     * @return все найденные в шарде джобы
     */
    public <A, R> List<DbQueueJob<A, R>> findFailedArchivedJobs(int shard, DbQueueJobType<A, R> jobType,
                                                                LocalDateTime from, LocalDateTime to) {
        Long jobTypeId = getJobTypeIdByJobType(jobType);

        var jobArchiveRecords = dslContextProvider.ppc(shard).select(JOB_ARCHIVE_FIELDS_TO_READ)
                .from(DBQUEUE_JOB_ARCHIVE)
                .where(DBQUEUE_JOB_ARCHIVE.JOB_TYPE_ID.eq(jobTypeId))
                .and(DBQUEUE_JOB_ARCHIVE.STATUS.eq(DbqueueJobArchiveStatus.Failed))
                .and(DBQUEUE_JOB_ARCHIVE.GRABBED_AT.ge(from).and(DBQUEUE_JOB_ARCHIVE.GRABBED_AT.lt(to)))
                .fetchInto(DBQUEUE_JOB_ARCHIVE);

        return StreamEx.of(jobArchiveRecords)
                .map(j -> modelFromJobArchiveTable(jobType, j))
                .collect(Collectors.toList());
    }

    public <A, R> DbQueueJob<A, R> insertJob(int shard, DbQueueJobType<A, R> jobType,
                                             ClientId clientId, long uid, A args) {
        return insertJob(shard, jobType, constructJob(clientId, uid, args));
    }

    public <A, R> List<DbQueueJob<A, R>> insertJobs(int shard, DbQueueJobType<A, R> jobType,
                                                    ClientId clientId, long uid, List<A> argsList) {
        List<DbQueueJob<A, R>> jobs = mapList(argsList, args -> constructJob(clientId, uid, args));
        return insertJobs(shard, jobType, jobs);
    }

    public <A, R> DbQueueJob<A, R> insertDelayedJob(int shard, DbQueueJobType<A, R> jobType,
                                                    ClientId clientId, long uid, A args, Duration delay) {
        LocalDateTime runAfter = now().plus(delay);
        return insertJob(shard, jobType, constructJob(clientId, uid, args, runAfter));
    }

    public <A, R> DbQueueJob<A, R> insertJob(DSLContext dslContext, String dbDescription,
                                             DbQueueJobType<A, R> jobType,
                                             ClientId clientId, long uid, A args) {
        return insertJob(dslContext, dbDescription, jobType, constructJob(clientId, uid, args));
    }

    public <A, R> DbQueueJob<A, R> insertJob(int shard, DbQueueJobType<A, R> jobType, DbQueueJob<A, R> job) {
        return insertJobs(shard, jobType, List.of(job)).get(0);
    }

    public <A, R> List<DbQueueJob<A, R>> insertJobs(int shard, DbQueueJobType<A, R> jobType,
                                                    List<DbQueueJob<A, R>> jobs) {
        Long jobTypeId = getJobTypeIdByJobType(jobType);
        DSLContext dslContext = dslContextProvider.ppc(shard);

        String dbDescription = "ppc:" + shard;

        return insertJobs(dslContext, dbDescription, jobTypeId, jobs);
    }

    public <A, R> DbQueueJob<A, R> insertJob(DSLContext dslContext, String dbDescription,
                                             DbQueueJobType<A, R> jobType, DbQueueJob<A, R> job) {
        Long jobTypeId = getJobTypeIdByJobType(jobType);

        return insertJob(dslContext, dbDescription, jobTypeId, job);
    }

    public <A, R> DbQueueJob<A, R> insertJob(DSLContext dslContext, String dbDescription, long jobTypeId,
                                             ClientId clientId, long uid, A args) {
        return insertJob(dslContext, dbDescription, jobTypeId, constructJob(clientId, uid, args));
    }

    public <A, R> DbQueueJob<A, R> insertJob(DSLContext dslContext, String dbDescription, Long jobTypeId,
                                             DbQueueJob<A, R> job) {
        return insertJobs(dslContext, dbDescription, jobTypeId, List.of(job)).get(0);
    }

    public <A, R> List<DbQueueJob<A, R>> insertJobs(DSLContext dslContext, String dbDescription, Long jobTypeId,
                                                    List<DbQueueJob<A, R>> jobs) {
        jobs.forEach(job -> checkArgument(job.getClientId() != null && job.getClientId().asLong() >= 0L,
                "job must have a ClientID, got %s", job.getClientId()));

        List<Long> jobIds = shardHelper.generateDbQueueJobIds(jobs.size());

        EntryStream.zip(jobIds, jobs).forKeyValue((jobId, job) -> {
            job.setId(jobId);
            job.setJobTypeId(jobTypeId);
            job.setStatus(DbQueueJobStatus.NEW);
        });

        LOGGER.debug("inserting jobs: {} into {}", jobs, dbDescription);

        InsertHelper<DbqueueJobsRecord> insertHelper = new InsertHelper<>(dslContext, DBQUEUE_JOBS);

        jobs.forEach(job -> insertHelper
                .add(JOBS_MAPPER, job)
                .set(DBQUEUE_JOBS.ARGS, compressArgs(job.getArgs()))
                .set(DBQUEUE_JOBS.CREATE_TIME, DSL.currentLocalDateTime())
                .newRecord());

        insertHelper.executeIfRecordsAdded();

        LocalDateTime now = LocalDateTime.now();
        jobs.forEach(job -> job.setCreationTime(now));

        return jobs;
    }


    @QueryWithoutIndex("Здесь индекс есть, подзапрос нужен т.к. jooq не умеет update + order")
    private int markJobsGrabbed(int shard, Duration grabDuration, Long jobTypeId, String globalId, int limit) {
        LOGGER.debug("trying to grab {} jobs in shard {}", limit, shard);

        int updatedRows = dslContextProvider.ppc(shard).update(DBQUEUE_JOBS)
                .set(DBQUEUE_JOBS.STATUS, DbqueueJobsStatus.Grabbed)
                .set(DBQUEUE_JOBS.GRABBED_BY, globalId)
                .set(DBQUEUE_JOBS.GRABBED_AT, DSL.currentLocalDateTime())
                .set(DBQUEUE_JOBS.GRABBED_UNTIL,
                        SqlUtils.localDateTimeAdd(DSL.currentLocalDateTime(), grabDuration))
                .set(DBQUEUE_JOBS.TRYCOUNT, DBQUEUE_JOBS.TRYCOUNT.add(1))

                // здесь подзапрос, потому что jOOQ не умеет orderBy и limit у update-запросов
                // здесь подзапрос с подзапросом, потому что подзапрос, который аргумент in(),
                // не может быть с limit - mysql говорит "This version of MySQL doesn't yet support <...>"
                .where(DBQUEUE_JOBS.JOB_ID
                        .in(DSL.select(DSL.field("jobId", Long.class)).from(DSL.select(DBQUEUE_JOBS.JOB_ID.as("jobId"))
                                .from(DBQUEUE_JOBS)
                                .where(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId))
                                .and(DBQUEUE_JOBS.NAMESPACE.isNull())
                                .and(DBQUEUE_JOBS.STATUS.eq(DbqueueJobsStatus.New)
                                        .or(DBQUEUE_JOBS.GRABBED_UNTIL.lessThan(DSL.currentLocalDateTime())))
                                .and(DBQUEUE_JOBS.RUN_AFTER.isNull()
                                        .or(DBQUEUE_JOBS.RUN_AFTER.lessThan(DSL.currentLocalDateTime())))
                                .orderBy(DBQUEUE_JOBS.JOB_ID)
                                .limit(limit))))
                .execute();

        LOGGER.debug("updated rows: {}", updatedRows);
        return updatedRows;
    }

    public <A, R> void markJobFinished(int shard, DbQueueJob<A, R> job, R result) {
        markJobDone(shard, job, DbQueueJobStatus.FINISHED, result);
    }

    private <A, R> void markJobDone(int shard, DbQueueJob<A, R> job, DbQueueJobStatus status, R result) {
        job.setStatus(status);
        job.setResult(result);

        LOGGER.debug("updating status of job #{} in shard {} to {}, result = {}", job.getId(), shard, status, result);

        byte[] packedResult = compressResult(result);

        for (int tries = 0; ; tries++) {
            try {
                dslContextProvider.ppcTransaction(shard, configuration -> {
                    DSLContext txContext = configuration.dsl();

                    txContext.insertInto(DBQUEUE_JOB_ARCHIVE, JOB_ARCHIVE_FIELDS_TO_FILL_WHEN_MARKING_DONE)
                            .select(DSL.select(JOBS_FIELDS_TO_COPY_TO_ARCHIVE)
                                    .select(DSL.val(status.toJobArchiveTableSource())
                                            .as(DBQUEUE_JOB_ARCHIVE.STATUS.getName()))
                                    .select(DSL.val(packedResult).as(DBQUEUE_JOB_ARCHIVE.RESULT.getName()))
                                    .from(DBQUEUE_JOBS)
                                    .where(DBQUEUE_JOBS.JOB_ID.eq(job.getId()))
                                    .forUpdate())
                            .execute();

                    txContext.deleteFrom(DBQUEUE_JOBS)
                            .where(DBQUEUE_JOBS.JOB_ID.eq(job.getId()))
                            .execute();
                });

                break;
            } catch (RuntimeException e) {
                LOGGER.error("error while trying to move job {} to archive (tries = {}): {}", job, tries, e);

                if (tries >= MAX_ATTEMPTS_TO_MOVE_TO_ARCHIVE) {
                    throw e;
                }
            }

        }
    }

    /**
     * пометить заданную джобу отмененной с переносом в архив
     *
     * @param job джоба, которую помечаем отмененной
     */
    public <A, R> void markJobRevoked(int shard, DbQueueJob<A, R> job, R result) {
        switch (job.getStatus()) {
            case REVOKED:
                break;
            case FINISHED:
                // у джобы со статусом FINISHED или FAILED просто меняем статус в БД
            case FAILED:
                dslContextProvider.ppc(shard).update(DBQUEUE_JOB_ARCHIVE)
                        .set(DBQUEUE_JOB_ARCHIVE.STATUS, DbqueueJobArchiveStatus.Revoked)
                        .set(DBQUEUE_JOB_ARCHIVE.RESULT, compressResult(result))
                        .where(DBQUEUE_JOB_ARCHIVE.JOB_ID.eq(job.getId()))
                        .execute();
                job.setStatus(DbQueueJobStatus.REVOKED);
                break;
            default:
                // джобу из таблицы dbqueue_jobs переносим в dbqueue_job_archive и меняем статус
                markJobDone(shard, job, DbQueueJobStatus.REVOKED, result);
        }
    }

    public <A, R> void markJobFailedPermanently(int shard, DbQueueJob<A, R> job, R error) {
        markJobDone(shard, job, DbQueueJobStatus.FAILED, error);
    }

    public <A, R> void markJobFailedOnce(int shard, DbQueueJob<A, R> job) {
        markJobFailedOnce(shard, job, null, null);
    }

    public <A, R> void markJobFailedOnce(int shard, DbQueueJob<A, R> job, @Nullable LocalDateTime runAfter) {
        markJobFailedOnce(shard, job, runAfter, null);
    }

    public <A, R> void markJobFailedOnce(int shard, DbQueueJob<A, R> job, @Nullable LocalDateTime runAfter,
                                         @Nullable A args) {
        job.setStatus(DbQueueJobStatus.NEW);
        job.setGrabbedUntil(null);
        job.setGrabbedBy("");

        LOGGER.debug("returning job #{} to the queue in shard {}", job.getId(), shard);

        var updateQuery = dslContextProvider.ppc(shard)
                .update(DBQUEUE_JOBS)
                .set(DBQUEUE_JOBS.STATUS, DbqueueJobsStatus.New)
                .set(DBQUEUE_JOBS.GRABBED_UNTIL, (LocalDateTime) null)
                .set(DBQUEUE_JOBS.RUN_AFTER, runAfter)
                .set(DBQUEUE_JOBS.GRABBED_BY, "");

        if (args != null) {
            LOGGER.debug("updating job args {}", job.getId());
            byte[] argsAsBytes = compressArgs(args);
            updateQuery.set(DBQUEUE_JOBS.ARGS, argsAsBytes);
        }

        updateQuery
                .where(DBQUEUE_JOBS.JOB_ID.eq(job.getId()))
                .execute();
    }

    /**
     * Обновить параметры джобы (полезно при презапуске, когда нужно продолжить выполнение)
     * Аргументы полностью перетираются в колонке переданным значением
     */
    public <A, R> void updateArgs(int shard, DbQueueJob<A, R> job, A args) {
        LOGGER.debug("updating job args {}", job.getId());

        byte[] argsAsBytes = compressArgs(args);

        dslContextProvider.ppc(shard)
                .update(DBQUEUE_JOBS)
                .set(DBQUEUE_JOBS.ARGS, argsAsBytes)
                .where(DBQUEUE_JOBS.JOB_ID.eq(job.getId()))
                .execute();
    }

    /**
     * удалить старые задачи из очереди и из архива
     *
     * @param duration    какие задачи считать старыми; "30 дней" значит "старше 30 дней"; "-30 дней" нельзя
     * @param limitOffset сколько максимум удалять из каждой таблицы; "10" значит "10 из jobs и 10 из archive", и
     *                    сколько задач пропустить от начала выборки
     * @return количество удалённых задач
     */
    public <A, R> int cleanup(int shard, DbQueueJobType<A, R> jobType, Duration duration, LimitOffset limitOffset) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        checkArgument(!duration.isNegative(), "must be positive duration: %s", duration);

        int deletedJobs = 0;

        Long jobTypeId = getJobTypeIdByJobType(jobType);

        List<DbQueueJob<A, R>> jobsInQueue = dslContext.select(JOBS_FIELDS_TO_READ)
                .from(DBQUEUE_JOBS)
                .where(DBQUEUE_JOBS.CREATE_TIME
                        .lessThan(SqlUtils.localDateTimeAdd(DSL.currentLocalDateTime(), duration.negated())))
                .and(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId))
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch()
                .into(DBQUEUE_JOBS)
                .map(record -> modelFromJobsTable(jobType, record));

        for (List<DbQueueJob<A, R>> subList : partition(jobsInQueue, DELETION_CHUNK_SIZE)) {
            for (DbQueueJob<A, R> job : subList) {
                LOGGER.debug("deleting job (in queue) {} from shard {}", job, shard);
            }

            deletedJobs = deletedJobs + dslContext.deleteFrom(DBQUEUE_JOBS)
                    .where(DBQUEUE_JOBS.JOB_ID.in(mapList(subList, DbQueueJob::getId)))
                    .execute();
        }

        List<DbQueueJob<A, R>> jobsInArchive = dslContext.select(JOB_ARCHIVE_FIELDS_TO_READ)
                .from(DBQUEUE_JOB_ARCHIVE)
                .where(DBQUEUE_JOB_ARCHIVE.CREATE_TIME
                        .lessThan(SqlUtils.localDateTimeAdd(DSL.currentLocalDateTime(), duration.negated())))
                .and(DBQUEUE_JOB_ARCHIVE.JOB_TYPE_ID.eq(jobTypeId))
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch()
                .into(DBQUEUE_JOB_ARCHIVE)
                .map(record -> modelFromJobArchiveTable(jobType, record));

        for (List<DbQueueJob<A, R>> subList : partition(jobsInArchive, DELETION_CHUNK_SIZE)) {
            for (DbQueueJob<A, R> job : subList) {
                LOGGER.debug("deleting job (in archive) {} from shard {}", job, shard);
            }

            deletedJobs = deletedJobs + dslContext.deleteFrom(DBQUEUE_JOB_ARCHIVE)
                    .where(DBQUEUE_JOB_ARCHIVE.JOB_ID.in(mapList(subList, DbQueueJob::getId)))
                    .execute();
        }

        return deletedJobs;
    }

    /**
     * удалить заданные джобы из архива
     *
     * @param ids    идентификаторы задач
     *
     * @return количество удалённых задач
     */
    public int deleteArchivedJobs(int shard, Set<Long> ids) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        return dslContext.deleteFrom(DBQUEUE_JOB_ARCHIVE)
                .where(DBQUEUE_JOB_ARCHIVE.JOB_ID.in(ids))
                .execute();
    }

    public <A, R> List<DbQueueJob<A, R>> getJobsByJobType(
            int shard,
            DbQueueJobType<A, R> jobType,
            LimitOffset limitOffset
    ) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        Long jobTypeId = getJobTypeIdByJobType(jobType);
        return dslContext
                .select(JOBS_FIELDS_TO_READ)
                .from(DBQUEUE_JOBS)
                .where(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId))
                .orderBy(DBQUEUE_JOBS.JOB_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch()
                .into(DBQUEUE_JOBS)
                .map(record -> modelFromJobsTable(jobType, record));
    }

    public <A, R> List<DbQueueJob<A, R>> getJobsByJobTypeAndClientIds(
            int shard,
            DbQueueJobType<A, R> jobType,
            Collection<Long> clientIds,
            LimitOffset limitOffset
    ) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        Long jobTypeId = getJobTypeIdByJobType(jobType);
        return dslContext
                .select(JOBS_FIELDS_TO_READ)
                .from(DBQUEUE_JOBS)
                .where(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId).and(DBQUEUE_JOBS.CLIENT_ID.in(clientIds)))
                .orderBy(DBQUEUE_JOBS.JOB_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch()
                .into(DBQUEUE_JOBS)
                .map(record -> modelFromJobsTable(jobType, record));
    }

    public <A, R> List<DbQueueJob<A, R>> getJobsFromArchiveByJobTypeAndClientIds(
            int shard,
            DbQueueJobType<A, R> jobType,
            Collection<Long> clientIds,
            LimitOffset limitOffset
    ) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        Long jobTypeId = getJobTypeIdByJobType(jobType);
        return dslContext
                .select(JOB_ARCHIVE_FIELDS_TO_READ)
                .from(DBQUEUE_JOB_ARCHIVE)
                .where(DBQUEUE_JOB_ARCHIVE.JOB_TYPE_ID.eq(jobTypeId).and(DBQUEUE_JOB_ARCHIVE.CLIENT_ID.in(clientIds)))
                .orderBy(DBQUEUE_JOB_ARCHIVE.JOB_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch()
                .into(DBQUEUE_JOB_ARCHIVE)
                .map(record -> modelFromJobArchiveTable(jobType, record));
    }

    @Nullable
    public <A, R> LocalDateTime getMinCreateTimeByJobType(int shard, DbQueueJobType<A, R> jobType) {
        Long jobTypeId = getJobTypeIdByJobType(jobType);
        Field<LocalDateTime> minCreateTime = DSL.min(DBQUEUE_JOBS.CREATE_TIME);
        return dslContextProvider.ppc(shard)
                .select(minCreateTime)
                .from(DBQUEUE_JOBS)
                .where(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId))
                .fetchOne(minCreateTime);
    }

    public <A, R> long getQueueSizeByJobType(int shard, DbQueueJobType<A, R> jobType) {
        Long jobTypeId = getJobTypeIdByJobType(jobType);
        return dslContextProvider.ppc(shard)
                .selectCount()
                .from(DBQUEUE_JOBS)
                .where(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId))
                .fetchOne(DSL.count());
    }

    public <A, R> long getClientsCountByJobType(int shard, DbQueueJobType<A, R> jobType) {
        Long jobTypeId = getJobTypeIdByJobType(jobType);
        return dslContextProvider.ppc(shard)
                .select(DSL.countDistinct(DBQUEUE_JOBS.CLIENT_ID))
                .from(DBQUEUE_JOBS)
                .where(DBQUEUE_JOBS.JOB_TYPE_ID.eq(jobTypeId))
                .fetchOne(DSL.count());
    }

    private <A, R> DbQueueJob<A, R> constructJob(ClientId clientId, long uid, A args) {
        return constructJob(clientId, uid, args, null);
    }

    private <A, R> DbQueueJob<A, R> constructJob(ClientId clientId,
                                                 long uid,
                                                 A args,
                                                 @Nullable LocalDateTime runAfter) {
        return new DbQueueJob<A, R>()
                .withClientId(clientId)
                .withUid(uid)
                .withArgs(args)
                .withRunAfter(runAfter)
                .withGrabbedBy("")
                .withPriority(0L)
                .withTryCount(0L)
                .withCreateTime(now());
    }

    private <A, R> Long getJobTypeIdByJobType(DbQueueJobType<A, R> jobType) {
        return typeMap.getJobTypeIdByTypeName(jobType.getName());
    }

    private static <T> byte[] compressArgs(T args) {
        return RepositoryUtils.toCompressedJsonDb(args, MAX_PACKED_ARGS_LENGTH);
    }

    private static <T> byte[] compressResult(T result) {
        return RepositoryUtils.toCompressedJsonDb(result, MAX_PACKED_RESULT_LENGTH);
    }

    private static <T> T uncompress(byte[] bytes, Class<T> cls) {
        return RepositoryUtils.objectFromCompressedJsonDb(bytes, cls);
    }
}
