package ru.yandex.direct.dbqueue.service;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dbqueue.DbQueueJobMaxAttemptCalculator;
import ru.yandex.direct.dbqueue.DbQueueJobMaxAttemptDefaultCalculator;
import ru.yandex.direct.dbqueue.DbQueueJobType;
import ru.yandex.direct.dbqueue.JobDelayedWithTryLaterException;
import ru.yandex.direct.dbqueue.JobFailedPermanentlyException;
import ru.yandex.direct.dbqueue.JobFailedWithTryLaterException;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;

import static com.google.common.base.Preconditions.checkNotNull;

@ParametersAreNonnullByDefault
public class DbQueueService {
    private final DbQueueRepository dbQueueRepository;

    private static final Logger LOGGER = LoggerFactory.getLogger(DbQueueService.class);

    public DbQueueService(DbQueueRepository dbQueueRepository) {
        this.dbQueueRepository = dbQueueRepository;
    }

    private <A, R> void processJobSafely(int shard, DbQueueJob<A, R> job,
                                         DbQueueJobType<A, R> jobType,
                                         Function<DbQueueJob<A, R>, R> jobProcessor,
                                         DbQueueJobMaxAttemptCalculator<A> maxAttemptsCalculator,
                                         BiFunction<DbQueueJob<A, R>, String, R> errorResultProducer) {
        R result = null;

        try {
            result = checkNotNull(jobProcessor.apply(job));
        } catch (RuntimeException e) {
            if (!(e instanceof JobDelayedWithTryLaterException)) {
                LOGGER.error("error while handling a job", e);
            }

            if (!(e instanceof JobFailedPermanentlyException) &&
                    job.getTryCount() < maxAttemptsCalculator.getMaxAttemptCount(job.getArgs())) {
                LocalDateTime runAfter = null;
                A argsToUpdate = null;
                if (e instanceof JobFailedWithTryLaterException) {
                    Duration tryLater = ((JobFailedWithTryLaterException) e).getTryLater();
                    runAfter = LocalDateTime.now().plusNanos(tryLater.toNanos());
                    argsToUpdate = castArguments(((JobFailedWithTryLaterException) e).getArgs(), jobType);
                }

                dbQueueRepository.markJobFailedOnce(shard, job, runAfter, argsToUpdate);
            } else {
                dbQueueRepository.markJobFailedPermanently(shard, job,
                        errorResultProducer.apply(job, ExceptionUtils.getStackTrace(e)));

            }
        }

        if (result != null) {
            dbQueueRepository.markJobFinished(shard, job, result);
        }
    }

    @Nullable
    @SuppressWarnings("unchecked")
    private static <A> A castArguments(@Nullable Object args, DbQueueJobType<A, ?> jobType) {
        if (args == null) {
            return null;
        }

        Class<A> argsClass = jobType.getArgsClass();
        if (!argsClass.isInstance(args)) {
            LOGGER.error("Cannot cast job args ({}) to class {}", args, argsClass);
            return null;
        }

        return (A) args;
    }

    public <A, R> boolean grabAndProcessJob(int shard,
                                            DbQueueJobType<A, R> jobType,
                                            Function<DbQueueJob<A, R>, R> jobProcessor,
                                            DbQueueJobMaxAttemptCalculator<A> maxAttemptsCalculator,
                                            BiFunction<DbQueueJob<A, R>, String, R> errorResultProducer) {
        return grabAndProcessJob(shard, jobType, DbQueueRepository.DEFAULT_GRAB_DURATION,
                jobProcessor, maxAttemptsCalculator, errorResultProducer);
    }

    public <A, R> boolean grabAndProcessJob(int shard,
                                            DbQueueJobType<A, R> jobType,
                                            Function<DbQueueJob<A, R>, R> jobProcessor,
                                            int maxAttempts,
                                            BiFunction<DbQueueJob<A, R>, String, R> errorResultProducer) {
        return grabAndProcessJob(shard, jobType, DbQueueRepository.DEFAULT_GRAB_DURATION,
                jobProcessor, maxAttempts, errorResultProducer);
    }

    public <A, R> boolean grabAndProcessJob(int shard,
                                            DbQueueJobType<A, R> jobType,
                                            Duration grabDuration,
                                            Function<DbQueueJob<A, R>, R> jobProcessor,
                                            int maxAttempts,
                                            BiFunction<DbQueueJob<A, R>, String, R> errorResultProducer) {
        DbQueueJob<A, R> job = dbQueueRepository.grabSingleJob(shard, jobType, grabDuration);

        if (job == null) {
            return false;
        }

        processJobSafely(shard, job, jobType, jobProcessor, new DbQueueJobMaxAttemptDefaultCalculator<>(maxAttempts),
                errorResultProducer);
        return true;
    }

    /**
     * Попробовать захватить задачу и один раз её обработать, по результатам обновить статус в базе
     *
     * @param jobProcessor        обработчик, вызывается, чтобы обработать задачу и отдать объект результата
     * @param errorResultProducer если не получилось, вызывается, чтобы сделать объект результата
     *                            из задачи и строки со стэктрейсом
     * @return нашлась ли задача для обработки
     */
    public <A, R> boolean grabAndProcessJob(int shard,
                                            DbQueueJobType<A, R> jobType,
                                            Duration grabDuration,
                                            Function<DbQueueJob<A, R>, R> jobProcessor,
                                            DbQueueJobMaxAttemptCalculator<A> maxAttemptsCalculator,
                                            BiFunction<DbQueueJob<A, R>, String, R> errorResultProducer) {
        DbQueueJob<A, R> job = dbQueueRepository.grabSingleJob(shard, jobType, grabDuration);

        if (job == null) {
            return false;
        }

        processJobSafely(shard, job, jobType, jobProcessor, maxAttemptsCalculator, errorResultProducer);
        return true;
    }


    /**
     * Зафейлить выполнение указанной задачи по типу и идентификатору
     */
    public <A, R> void markJobFailedOnce(int shard, DbQueueJobType<A, R> jobType, Long jobId) {
        markJobFailedOnceRunAfter(shard, jobType, jobId, null);
    }


    /**
     * Зафейлить выполнение указанной задачи по типу и идентификатору, указать период времени через сколько попробовать выполнить повторно.
     */
    public <A, R> void markJobFailedOnce(int shard, DbQueueJobType<A, R> jobType, Long jobId, Duration tryInDuration) {
        LocalDateTime runAfter = LocalDateTime.now().plusNanos(tryInDuration.toNanos());
        markJobFailedOnceRunAfter(shard, jobType, jobId, runAfter);
    }


    /**
     * Зафейлить выполнение указанной задачи по типу и идентификатору, выставить время после которого можно попробовать выполнить повторно.
     */
    private <A, R> void markJobFailedOnceRunAfter(int shard, DbQueueJobType<A, R> jobType, Long jobId,
                                                  @Nullable LocalDateTime runAfter) {
        DbQueueJob<A, R> job = dbQueueRepository.findJobById(shard, jobType, jobId);
        if (job == null) {
            LOGGER.warn("job with type {} and id {} not found in {} shard. Cannot fail lost job.",
                    jobType.getName(), jobId, shard);
            return;
        }

        dbQueueRepository.markJobFailedOnce(shard, job, runAfter);
    }
}
