package ru.yandex.direct.dbutil;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.function.ToDoubleFunction;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Triple;

@ParametersAreNonnullByDefault
public class CollectionUtils {
    /**
     * Сгенерировать новый список на основе заданной коллекции перемешенный с учетом весов элеметов
     *
     * @param random   Генератор случайных чисел
     * @param coll     Коллекция
     * @param weighter Функция взвешивающая элементы. Должна возвращать неотрицательные значения
     */
    @Nonnull
    public static <T> List<T> weightedShuffle(
            Random random, Collection<T> coll, ToDoubleFunction<? super T> weighter) {
        Objects.requireNonNull(random, "random");
        Objects.requireNonNull(coll, "coll");
        Objects.requireNonNull(weighter, "weighter");

        return coll.stream()
                .map(el -> {
                    double weight = weighter.applyAsDouble(el);
                    return Triple.of(
                            el,
                            weight > 0 ? -Math.log(random.nextDouble()) / weight : Double.POSITIVE_INFINITY,
                            random.nextDouble());
                })
                .sorted(
                        Comparator.comparing(Triple<T, Double, Double>::getMiddle)
                                .thenComparing(Triple<T, Double, Double>::getRight))
                .map(Triple::getLeft)
                .collect(Collectors.toList());
    }
}
