package ru.yandex.direct.dbutil;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;

/**
 * Класс для формирования условия запроса к БД по классу-критерию.
 * Обрабатывает только заданные (не {@code null}) значения, запоминает наличие пустых условий.
 */
@ParametersAreNonnullByDefault
public class ConditionsAccumulator {
    private final List<Condition> conditions;
    private boolean hasEmptyCondition;

    public ConditionsAccumulator() {
        this.conditions = new ArrayList<>();
    }

    /**
     * Добавляет в результат условие на критерий (преобразованный конвертером), если он не {@code null}.
     * <p>
     * Запоминает, если условие - это пустая коллекция.
     * Используйте это признак с осторожностью при отрицательных условиях (например notIn)
     *
     * @param criteriaSupplier  источник данных для условия
     * @param converter         метод приведения типов
     * @param conditionFunction метод, дающий условие на колонку БД
     * @param <C>               тип критерия
     * @param <R>               тип условия БД
     */
    public <C, R> void add(Supplier<C> criteriaSupplier,
                           Function<C, R> converter,
                           Function<R, Condition> conditionFunction) {
        C criteria = criteriaSupplier.get();
        if (criteria == null) {
            return;
        }
        if (criteria instanceof Collection && ((Collection) criteria).isEmpty()) {
            hasEmptyCondition = true;
        }
        conditions.add(conditionFunction.apply(converter.apply(criteria)));
    }

    /**
     * Получить список условий к БД
     */
    public List<Condition> getConditions() {
        return conditions;
    }

    /**
     * Есть ли среди условий - вырожденные
     */
    public boolean containsEmptyCondition() {
        return hasEmptyCondition;
    }

    /**
     * Было ли добавлено хотя бы одно условие?
     */
    public boolean containsConditions() {
        return !conditions.isEmpty();
    }
}
