package ru.yandex.direct.dbutil;

import java.sql.Timestamp;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Iterator;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.jooq.Condition;
import org.jooq.DatePart;
import org.jooq.Field;
import org.jooq.InsertOnDuplicateSetMoreStep;
import org.jooq.InsertOnDuplicateStep;
import org.jooq.InsertValuesStep1;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.SelectField;
import org.jooq.impl.DSL;
import org.jooq.types.DayToSecond;
import org.jooq.util.mysql.MySQLDSL;

import ru.yandex.direct.dbutil.field.SetField;

@ParametersAreNonnullByDefault
public class SqlUtils {
    private static final String EMPTY_STRING = "";
    public static final Long ID_NOT_SET = 0L;
    public static final String STRAIGHT_JOIN = "STRAIGHT_JOIN";
    public static final String PRIMARY = "PRIMARY";

    /**
     * NULL, можно использовать в ORDER BY
     */
    public static final Field<Byte> NULL_FILED = DSL.inline((Byte) null);

    /**
     * Дефолтное значение количества записей при разбиении запроса на части
     */
    public static final int TYPICAL_SELECT_CHUNK_SIZE = 10000;

    /**
     * Нулевой mysql-timestamp
     */
    public static final String ZERO_TIMESTAMP = "0000-00-00 00:00:00";

    /**
     * Фабричный метод для создания {@link SetField} &ndash; обёртки для поля с {@code SET}-специфичными методами.
     *
     * @param field {@code SET} поле
     * @return экземпляр {@link SetField}
     */
    @Nonnull
    public static SetField setField(SelectField<String> field) {
        return new SetField(field);
    }

    /**
     * (частичная) замена mysql-оператора replace. Генерирует конструкцию вида
     * <pre>
     *     <code>
     *     ON DUPLICATE KEY UPDATE
     *       value1 = VALUES(value1),
     *       value2 = VALUES(value2),
     *       ...
     *       valueN = values(valueN)
     *     </code>
     * </pre>
     * Полная замена затруднительна ввиду того, что {@link InsertOnDuplicateStep} можно получить
     * только из {@link InsertValuesStep1#select(Select)}...{@link org.jooq.InsertValuesStep22#select(Select)},
     * но не из {@link org.jooq.InsertValuesStepN#select(Select)}
     *
     * @param query  - настраиваемый query
     * @param fields - поля, которые требуется обновить
     * @return объект класса InsertOnDuplicateSetMoreStep, если {@code fields} не пуст, иначе null
     * @throws IllegalArgumentException если fields пустой
     */
    @Nonnull
    public static <T, R extends Record> InsertOnDuplicateSetMoreStep<R> onConflictUpdate(
            InsertOnDuplicateStep<R> query,
            Iterable<Field<T>> fields) {
        Iterator<? extends Field<T>> fieldsIter = fields.iterator();
        Preconditions.checkArgument(fieldsIter.hasNext(), "can't produce sql for empty fields");
        // field нужен до цикла, чтобы  возвращаемый тип был InsertOnDuplicateSetMoreStep
        // такой тип вернет InsertOnDuplicateSetStep.set
        Field<T> field = fieldsIter.next();
        Field<T> fieldValue = MySQLDSL.values(field);
        InsertOnDuplicateSetMoreStep<R> stepSet = query
                .onDuplicateKeyUpdate()
                .set(field, fieldValue);
        while (fieldsIter.hasNext()) {
            field = fieldsIter.next();
            fieldValue = MySQLDSL.values(field);
            stepSet = stepSet.set(field, fieldValue);
        }
        return stepSet;
    }

    /**
     * @return 'нулевую' mysql-дату '0000-00-00'
     */
    public static Field<java.sql.Date> mysqlZeroDate() {
        return DSL.value("0000-00-00").cast(java.sql.Date.class);
    }

    /**
     * @return 'нулевую' mysql-дату '0000-00-00'
     */
    public static Field<LocalDate> mysqlZeroLocalDate() {
        return DSL.value("0000-00-00").cast(LocalDate.class);
    }

    /**
     * @return 'нулевой' mysql-timestamp '0000-00-00 00:00:00'
     */
    public static Field<java.sql.Timestamp> mysqlZeroTimestamp() {
        return DSL.value(ZERO_TIMESTAMP).cast(java.sql.Timestamp.class);
    }

    /**
     * @return 'нулевой' mysql-localdatetime '0000-00-00 00:00:00'
     */
    public static Field<LocalDateTime> mysqlZeroLocalDateTime() {
        return DSL.field("{0}", LocalDateTime.class, ZERO_TIMESTAMP);
    }

    /**
     * mysql-ная функция UTC_TIMESTAMP()
     *
     * @see <a href="https://dev.mysql.com/doc/refman/5.7/en/date-and-time-functions.html#function_utc-timestamp">
     * Mysql server documentation
     * </a>
     */
    public static Field<Timestamp> mysqlUtcTimestamp() {
        return DSL.field("UTC_TIMESTAMP()", Timestamp.class);
    }

    /**
     * MysqlDSL - преобразование времени из UTC в таймзону сервера
     */
    public static Field<Timestamp> mysqlConvertFromUtc(Field<Timestamp> value) {
        return DSL.field("CONVERT_TZ({0}, {1}, {2})", Timestamp.class, value, mysqlUtcTz(), mysqlLocalTz());
    }

    private static Field<String> mysqlUtcTz() {
        return DSL.field("'+00:00'", String.class);
    }

    private static Field<String> mysqlLocalTz() {
        return DSL.field("@@time_zone", String.class);
    }

    /**
     * Типобезопасная обвязка к mysql-функции unix_timestamp:
     *
     * @see <a href="https://dev.mysql.com/doc/refman/5.7/en/date-and-time-functions.html#function_unix-timestamp">
     * UNIX_TIMESTAMP(date)
     * </a>
     */
    public static Field<Long> mysqlUnixTimestampForDate(Field<java.sql.Date> srcField) {
        return mysqlUnixTimestampInternal(srcField);
    }

    public static Field<Long> mysqlUnixTimestampForLocalDate(Field<LocalDate> srcField) {
        return mysqlUnixTimestampInternal(srcField);
    }

    /**
     * Типобезопасная обвязка к mysql-функции unix_timestamp:
     *
     * @see <a href="https://dev.mysql.com/doc/refman/5.7/en/date-and-time-functions.html#function_unix-timestamp">
     * UNIX_TIMESTAMP(date)
     * </a>
     */
    public static Field<Long> mysqlUnixTimestampForTimestamp(Field<java.sql.Timestamp> srcField) {
        return mysqlUnixTimestampInternal(srcField);
    }

    private static Field<Long> mysqlUnixTimestampInternal(Field<?> srcField) {
        return DSL.field("UNIX_TIMESTAMP({0})", Long.class, srcField);
    }

    /**
     * Типобезопасная обвязка к mysql-функции unix_timestamp:
     *
     * @see <a href="https://dev.mysql.com/doc/refman/5.7/en/date-and-time-functions.html#function_unix-timestamp">
     * UNIX_TIMESTAMP()
     * </a>
     */
    public static Field<Long> mysqlUnixTimestamp() {
        return DSL.field("UNIX_TIMESTAMP()", Long.class);
    }

    /**
     * Обвязка к timestampdiff с указанием DatePart
     * Возможно появится в будущих версиях jooq-а: https://github.com/jOOQ/jOOQ/issues/4303
     */
    public static Field<Integer> timestampDiff(DatePart part, Field<Timestamp> t1, Timestamp t2) {
        return DSL.field("TIMESTAMPDIFF({0}, {1}, {2})", Integer.class, DSL.keyword(part.toSQL()), t1, t2);
    }

    public static Field<Integer> localDateTimeDiff(DatePart part, Field<LocalDateTime> t1, LocalDateTime t2) {
        return DSL.field("TIMESTAMPDIFF({0}, {1}, {2})", Integer.class, DSL.keyword(part.toSQL()), t1, t2);
    }

    public static Field<LocalDateTime> localDateTimeAdd(Field<LocalDateTime> date, Duration dur) {
        return DSL.field("DATE_ADD({0}, INTERVAL {1} SECOND)", LocalDateTime.class, date, dur.getSeconds());
    }

    public static Field<LocalDateTime> localDateTimeAddMonth(Field<LocalDateTime> date, int monthCount) {
        return DSL.field("DATE_ADD({0}, INTERVAL {1} MONTH)", LocalDateTime.class, date, monthCount);
    }

    public static Field<Integer> mysqlGetLock(String lockName, Duration timeout) {
        return DSL.field("GET_LOCK({0}, {1})", Integer.class, lockName, timeout.getSeconds());
    }

    public static Field<Integer> mysqlReleaseLock(String lockName) {
        return DSL.field("RELEASE_LOCK({0})", Integer.class, lockName);
    }

    public static Field<Long> findInSet(String value, Field<?> column) {
        return DSL.function("FIND_IN_SET", Long.class, DSL.val(value), column);
    }

    /**
     * Перевести в объект длительности, отбросив милли- и наносекунды.
     */
    public static Duration convertToDuration(DayToSecond dayToSecond) {
        return Duration.ofDays(dayToSecond.getDays())
                .plusHours(dayToSecond.getHours())
                .plusMinutes(dayToSecond.getMinutes())
                .plusSeconds(dayToSecond.getSeconds())
                .multipliedBy(dayToSecond.getSign());
    }

    public enum SortOrder {
        ASC,
        DESC
    }

    public static <T> Field<T> fieldFromJson(Field<?> field, String name, Class<T> fieldClass) {
        return DSL.field("JSON_EXTRACT({0}, {1})", fieldClass, field, DSL.val("$." + name));
    }

    public static Condition isEmptyString(Field<String> field) {
        return field.isNull().or(field.eq(EMPTY_STRING));
    }
}
