package ru.yandex.direct.dbutil.configuration;

import java.util.HashMap;
import java.util.Map;

import com.typesafe.config.ConfigObject;
import com.typesafe.config.ConfigValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceWatcher;
import ru.yandex.direct.liveresource.LiveResourceWatcherFactory;
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.singletonList;

/**
 * Configuration for SQL-database specific utilities of Direct
 */
@Configuration
@Import(EssentialConfiguration.class)
@ComponentScan(
        basePackages = "ru.yandex.direct.dbutil",
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
)
public class DbUtilConfiguration {

    /**
     * Явные имена даются бинам для возможности их переопределения в
     * другом @Configuration-классе, который импортирует данный
     */
    public static final String DB_CONFIG_FACTORY = "dbConfigFactory";

    private static final Logger logger = LoggerFactory.getLogger(DbUtilConfiguration.class);

    DirectConfig directConfig;

    @Autowired
    public DbUtilConfiguration(DirectConfig directConfig) {
        this.directConfig = directConfig;
    }

    @Bean(name = DB_CONFIG_FACTORY)
    DbConfigFactory dbConfigFactory(LiveResourceFactoryBean liveResourceFactoryBean,
                                    LiveResourceWatcherFactory liveResourceWatcherFactory) {
        String dbConfig = directConfig.getString("db_config");
        logger.debug("read {}", dbConfig);

        // часть настроек подключений к базам данных, (которые не обновляются в реалтайме
        // и которые по этой причине не имеет смысла добавлять в "настоящий" db-config в zookeeper),
        // хранятся прямо в конфиге приложения - мы парсим эту секцию и добавляем её в DbConfigFactory
        Map<String, DbConfig> dbConfigOverrides = parseDbConfigOverridesMap(liveResourceFactoryBean);

        LiveResource liveResource = liveResourceFactoryBean.get(dbConfig);
        return createAndWatch(liveResource, liveResourceWatcherFactory, dbConfigOverrides);
    }

    private Map<String, DbConfig> parseDbConfigOverridesMap(LiveResourceFactoryBean liveResourceFactoryBean) {
        Map<String, DbConfig> map = new HashMap<>();
        ConfigObject overridesConfig = directConfig.getConfig().getObject("db_config_overrides");
        for (Map.Entry<String, ConfigValue> entry : overridesConfig.entrySet()) {
            String dbname = entry.getKey();
            ConfigValue config = entry.getValue();
            if (!(config instanceof ConfigObject)) {
                continue;
            }
            ConfigObject configObject = ((ConfigObject) config);
            DbConfig databaseConfig = parseDbConfigFromConfigBranch(
                    dbname,
                    new DirectConfig(configObject.toConfig()),
                    liveResourceFactoryBean
            );
            map.put(dbname, databaseConfig);
        }
        return map;
    }

    private DbConfig parseDbConfigFromConfigBranch(String dbname, DirectConfig branch,
                                                   LiveResourceFactoryBean liveResourceFactoryBean) {
        DbConfig dbConfig = new DbConfig();
        dbConfig.setDbName(dbname);

        if (branch.hasPath("port")) {
            dbConfig.setPort(branch.getInt("port"));
        }
        if (branch.hasPath("user")) {
            dbConfig.setUser(branch.getString("user"));
        }
        if (branch.hasPath("pass")) {
            dbConfig.setPass(liveResourceFactoryBean.get(branch.getString("pass")).getContent().trim());
        }
        if (branch.hasPath("engine")) {
            dbConfig.setEngine(DbConfig.Engine.valueOf(branch.getString("engine").toUpperCase()));
        }
        if (branch.hasPath("db")) {
            dbConfig.setDb(branch.getString("db"));
        }
        if (branch.hasPath("weight")) {
            dbConfig.setWeight(branch.getInt("weight"));
        }
        if (branch.hasPath("connect_timeout")) {
            dbConfig.setConnectTimeout(branch.getDouble("connect_timeout"));
        }
        if (branch.hasPath("request_retries")) {
            dbConfig.setRequestRetries(branch.getInt("request_retries"));
        }
        if (branch.hasPath("host")) {
            dbConfig.setHosts(singletonList(branch.getString("host")));
        }
        if (branch.hasPath("ssl")) {
            dbConfig.setSsl(branch.getBoolean("ssl"));
        }
        if (branch.hasPath("verify_ssl_certs")) {
            dbConfig.setVerify(branch.getBoolean("verify_ssl_certs"));
        }
        if (branch.hasPath("compression")) {
            dbConfig.setCompression(branch.getBoolean("compression"));
        }
        if (branch.hasPath("max_block_size")) {
            dbConfig.setClickhouseMaxBlockSize(branch.getInt("max_block_size"));
        }

        return dbConfig;
    }

    public static DbConfigFactory createAndWatch(LiveResource liveResource,
                                                 LiveResourceWatcherFactory liveResourceWatcherFactory,
                                                 Map<String, DbConfig> staticDbConfigs) {
        checkNotNull(liveResource, "liveResource is required");

        LiveResourceWatcher resourceWatcher = liveResourceWatcherFactory.createWatcher(liveResource);
        DbConfigFactory result = new DbConfigFactory(resourceWatcher.watch(), staticDbConfigs);
        resourceWatcher.addListener(result);
        return result;
    }
}
