package ru.yandex.direct.dbutil.sharding;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.ShardByClient;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Top-level methods over ShardSupport
 */
@Component
public class ShardHelper implements ShardByClient {

    private final ShardSupport shardSupport;

    @Autowired
    public ShardHelper(ShardSupport shardSupport) {
        Assert.notNull(shardSupport, "shardSupport is required");
        this.shardSupport = shardSupport;
    }

    /**
     * Получить рандомный шард
     */
    public int getRandomShard() {
        int first = 1;
        return dbShards().stream()
                .skip(RandomUtils.nextInt(first, dbShards().size()))
                .findFirst()
                .orElse(first);
    }

    /**
     * @see ShardSupport#getAvailablePpcShards()
     */
    public List<Integer> dbShards() {
        return shardSupport.getAvailablePpcShards();
    }

    /**
     * сгруппировать данные по шардам
     *
     * @param data         список элементов
     * @param key          ключ, по которому будет осуществляться поиск шарда в ppcdict, будет передан в
     *                     {@link ShardSupport#getShards(ShardKey, List)}
     * @param keyExtractor как получить значение ключа из одного элемена списка
     * @param <T>          класс элемента списка
     * @param <R>          класс ключа для {@link ShardSupport#getShards(ShardKey, List)}
     * @return {@link ShardedData}
     */
    public <T, R> ShardedData<T> groupByShard(Collection<T> data, ShardKey key, Function<T, R> keyExtractor) {
        if (!(data instanceof List)) {
            // для сопоставления данных с ключами нам важен порядок, поэтому,
            // если входные данные не список -- делаем список
            data = new ArrayList<>(data);
        }
        List<R> keys = mapList(data, keyExtractor);
        return new ShardedData<>(StreamEx.of(shardSupport.getShards(key, keys))
                .zipWith(data.stream())
                .filterKeys(Objects::nonNull)
                .grouping());
    }


    /**
     * применяет consumer для всех номеров шардов
     *
     * @param consumer consumer на вход получает номер шарда
     */
    public void forEachShard(Consumer<Integer> consumer) {
        try (TraceProfile ignored = Trace.current().profile("forEachShard")) {
            for (Integer dbShard : dbShards()) {
                consumer.accept(dbShard);
            }
        }
    }

    /**
     * Применить функцию для всех шардов параллельно, используя переданый {@code executorService}.
     *
     * @param function        функция для выполнения, принимающая номер шарда
     * @param <R>             тип результата, возвращаемого функцией
     * @param executorService сервис который будет выполнять запуск функции в каждом из шардов
     * @return словарь соответствий "номер щарда" - "результат выполнения функции в этом шарде"
     */
    public <R> Map<Integer, R> forEachShardParallel(Function<Integer, R> function, ExecutorService executorService) {
        Map<Integer, R> result = new ConcurrentHashMap<>();
        Collection<Integer> dbShards = dbShards();
        CompletableFuture[] futures = new CompletableFuture[dbShards.size()];
        int idx = 0;
        for (Integer dbShard : dbShards) {
            Supplier<R> resultProvider = () -> function.apply(dbShard);
            Consumer<R> resultConsumer = x -> result.put(dbShard, x);
            futures[idx++] = CompletableFuture.supplyAsync(resultProvider, executorService).thenAccept(resultConsumer);
        }
        try (TraceProfile ignored = Trace.current().profile("forEachShardParallel:join", "", futures.length)) {
            CompletableFuture.allOf(futures).join(); // IGNORE-BAD-JOIN DIRECT-149116
        }
        return result;
    }

    /**
     * Применить функцию для всех шардов последовательно.
     *
     * @param function функция для выполнения, принимающая номер шарда
     * @param <R>      тип результата, возвращаемого функцией
     * @return словарь соответствий "номер щарда" - "результат выполнения функции в этом шарде"
     */
    public <R> Map<Integer, R> forEachShardSequential(Function<Integer, R> function) {
        try (TraceProfile ignored = Trace.current().profile("forEachShardSequential")) {
            return listToMap(dbShards(), Function.identity(), function);
        }
    }

    public <T> ShardedData<T> groupByShard(Collection<T> data, ShardKey key) {
        return groupByShard(data, key, Function.identity());
    }

    public int getShardByClientUid(Long uid) {
        return shardSupport.getShard(ShardKey.UID, uid);
    }

    public int getShardByClientUidStrictly(Long uid) {
        int shard = getShardByClientUid(uid);
        Assert.state(shard != NO_SHARD, "can not find shard by uid = " + uid);
        return shard;
    }

    public int getShardByClientId(ClientId clientId) {
        return shardSupport.getShard(ShardKey.CLIENT_ID, clientId.asLong());
    }

    public int getShardByBannerId(Long bannerId) {
        return shardSupport.getShard(ShardKey.BID, bannerId);
    }

    /**
     * Получение шардов для баннеров, сгруппированных по шардам.
     * В возвращаемой мапе не будет айдишников, которых нет в базе.
     */
    @QueryWithForbiddenShardMapping("bid")
    public Map<Integer, List<Long>> getBannerIdsByShard(List<Long> bannerIds) {
        List<Integer> shards = shardSupport.getShards(ShardKey.BID, bannerIds);
        return EntryStream.of(bannerIds)
                .mapKeys(shards::get)
                .nonNullKeys()
                .grouping();
    }

    public int getShardByGroupId(Long adGroupId) {
        return shardSupport.getShard(ShardKey.PID, adGroupId);
    }

    public List<Integer> getShardsByGroupId(List<Long> adGroupIds) {
        return shardSupport.getShards(ShardKey.PID, adGroupIds);
    }

    /**
     * Получение списка шардов для списка клиентов
     * Если клиент не найден, то значение будет null
     */
    public Map<Long, Integer> getShardsByClientIds(List<Long> clientIds) {
        List<Integer> shards = shardSupport.getShards(ShardKey.CLIENT_ID, clientIds);
        Map<Long, Integer> result = new HashMap<>();
        for (int i = 0; i < shards.size(); i++) {
            result.put(clientIds.get(i), shards.get(i));
        }
        return result;
    }

    public int getShardByUserId(Long userId) {
        return shardSupport.getShard(ShardKey.UID, userId);
    }

    public int getShardByClientIdStrictly(ClientId clientId) {
        int shard = getShardByClientId(clientId);
        Assert.state(shard != NO_SHARD, "can not find shard by clientId = " + clientId);
        return shard;
    }

    public int getShardByLoginStrictly(String login) {
        int shard = shardSupport.getShard(ShardKey.LOGIN, login);
        Assert.state(shard != NO_SHARD, "can not find shard by login = " + login);
        return shard;
    }

    /**
     * Получить шард по номеру кампании по метабазе
     *
     * @param campaignId номер кампании
     */
    public int getShardByCampaignId(Long campaignId) {
        return shardSupport.getShard(ShardKey.CID, campaignId);
    }

    @Nullable
    public Long getUidByLogin(String login) {
        return shardSupport.getValue(ShardKey.LOGIN, login, ShardKey.UID, Long.class);
    }

    public List<List<String>> getLoginsByUids(List<?> uids) {
        return shardSupport.lookupKeysWithValues(ShardKey.LOGIN, ShardKey.UID, uids, String.class);
    }

    public List<Long> getUidsByLogin(List<String> logins) {
        return shardSupport.getValues(ShardKey.LOGIN, logins, ShardKey.UID, Long.class);
    }

    public Long getClientIdByAdGroupId(Long adGroupId) {
        return shardSupport.getValue(ShardKey.PID, adGroupId, ShardKey.CLIENT_ID, Long.class);
    }

    public List<Long> getClientIdsByBannerIds(List<Long> bannerIds) {
        return shardSupport.getValues(ShardKey.BID, bannerIds, ShardKey.CLIENT_ID, Long.class);
    }

    /**
     * Get Login for specified uid
     *
     * @param uid uid
     * @return corresponding Login
     * @throws IllegalArgumentException if no data found
     */
    public String getLoginByUid(long uid) {
        return getLoginsByUids(Collections.singletonList(uid))
                .get(0)
                .stream()
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("Login for uid = " + uid + " is not found"));
    }

    /**
     * Get map from campaign ids to client ids
     *
     * @param cids Desired campaign ids
     * @return Map with size less or equal to size of requested campaign ids
     */
    public Map<Long, Long> getClientIdsByCampaignIds(Collection<Long> cids) {
        List<Long> uniqueCids;
        if (cids instanceof Set) {
            uniqueCids = new ArrayList<>(cids);
        } else {
            uniqueCids = new ArrayList<>(new HashSet<>(cids));
        }
        List<Long> clientIds = shardSupport.getValues(ShardKey.CID, uniqueCids, ShardKey.CLIENT_ID, Long.class);
        return StreamEx.of(uniqueCids)
                .zipWith(StreamEx.of(clientIds))
                .filterValues(Objects::nonNull)
                .toMap();
    }

    /**
     * Get map from ad group ids to client ids
     *
     * @param pids Desired ad group ids
     * @return Map with size less or equal to size of requested ad group ids
     */
    public Map<Long, Long> getClientIdsByAdGroupIds(Collection<Long> pids) {
        List<Long> uniquePids = List.copyOf(Set.copyOf(pids));
        List<Long> clientIds = shardSupport.getValues(ShardKey.PID, uniquePids, ShardKey.CLIENT_ID, Long.class);
        return StreamEx.of(uniquePids)
                .zipWith(StreamEx.of(clientIds))
                .filterValues(Objects::nonNull)
                .toMap();
    }

    /**
     * Get ClientID for specified uid
     *
     * @param uid uid
     * @return corresponding ClientId
     * @throws IllegalArgumentException if no data found
     */
    public Long getClientIdByUid(long uid) {
        Long clientId = shardSupport.getValue(ShardKey.UID, uid, ShardKey.CLIENT_ID, Long.class);
        checkArgument(clientId != null, "Client for uid " + uid + " is not found");
        return clientId;
    }

    /**
     * Get ClientID for specified login
     *
     * @param login login
     * @return corresponding ClientId
     * @throws IllegalArgumentException if no data found
     */
    public Long getClientIdByLogin(String login) {
        Long clientId = shardSupport.getValue(ShardKey.LOGIN, login, ShardKey.CLIENT_ID, Long.class);
        checkArgument(clientId != null, "Client for login " + login + " is not found");
        return clientId;
    }

    /**
     * Аналог getClientIdByUid для списка uid-ов
     *
     * @param uids коллекция uid, для которых нужно получить clientId
     * @return - отображение uid -> clientId, uid-ы с отсутствующим clientId отфильтрованы
     */
    @Nonnull
    public Map<Long, Long> getClientIdsByUids(Collection<Long> uids) {
        List<Long> orderedUids;
        if (uids instanceof List) {
            orderedUids = (List<Long>) uids;
        } else {
            // для сопоставления clientId с uid нам важен порядок, поэтому,
            // если входные данные не список -- делаем список
            orderedUids = new ArrayList<>(uids);
        }
        List<Long> clientIds = shardSupport.getValues(ShardKey.UID, orderedUids, ShardKey.CLIENT_ID, Long.class);
        return StreamEx.of(uids)
                .zipWith(StreamEx.of(clientIds))
                .filterValues(Objects::nonNull)
                .distinctKeys()
                .toMap();
    }

    @Nonnull
    public Map<String, Long> getClientIdsByLogins(Collection<String> logins) {
        List<Long> uids = shardSupport.getValues(ShardKey.LOGIN, logins, ShardKey.CLIENT_ID, Long.class);
        return StreamEx.of(logins)
                .zipWith(StreamEx.of(uids))
                .filterValues(Objects::nonNull)
                .distinctKeys()
                .toMap();
    }

    /**
     * Получить список представителей клиента
     *
     * @param clientId идентификатор клиента
     * @return - список uid-ов представителей
     */
    public List<Long> getUidsByClientId(long clientId) {
        List<Long> uids = shardSupport.lookupKeysWithValue(ShardKey.UID, ShardKey.CLIENT_ID, clientId, Long.class);
        checkArgument(!uids.isEmpty(), "Uids for client " + clientId + " is not found");
        return uids;
    }

    public long getClientIdByCampaignId(long campaignId) {
        Long clientId = shardSupport.getValue(ShardKey.CID, campaignId, ShardKey.CLIENT_ID, Long.class);
        checkArgument(clientId != null, "Client for campaignId = " + campaignId + " is not found");
        return clientId;
    }

    public Long getMaxAdGroupId() {
        return shardSupport.getMaxValue(ShardKey.PID);
    }


    public List<Long> generateCampaignIds(long clientId, int count) {
        return longValues(shardSupport.generateValues(ShardKey.CID, ShardKey.CLIENT_ID, repeat(clientId, count)));
    }

    public List<Long> generateStrategyIds(long clientId, int count) {
        return longValues(shardSupport.generateValues(ShardKey.STRATEGY_ID, ShardKey.CLIENT_ID, repeat(clientId, count)));
    }

    public List<Long> generateAdGroupIds(long clientId, int count) {
        return longValues(shardSupport.generateValues(ShardKey.PID, ShardKey.CLIENT_ID, repeat(clientId, count)));
    }

    public List<Long> generateIdsByShardKey(ShardKey key, long clientId, int count) {
        return longValues(shardSupport.generateValues(key, ShardKey.CLIENT_ID, repeat(clientId, count)));
    }

    public List<Long> generateIdsByAutoIncrementKey(AutoIncrementKey key, int count) {
        return longValues(shardSupport.generateValues(key, count));
    }

    public List<Long> generateBannerIds(List<Long> groupIds) {
        return longValues(shardSupport.generateValues(ShardKey.BID, ShardKey.PID, groupIds));
    }

    public List<Long> generateBannerIdsByBids(List<Long> bannerIds) {
        return longValues(shardSupport.generateValues(ShardKey.BID, ShardKey.BID, bannerIds));
    }

    public List<Long> generateAdditionItemIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.ADDITIONS_ITEM_ID, count));
    }

    public List<Long> generateSitelinkSetIds(long clientId, int size) {
        return longValues(shardSupport.generateValues(ShardKey.SITELINKS_SET_ID, ShardKey.CLIENT_ID,
                repeat(clientId, size)));
    }

    public List<Long> generateSitelinkIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.SL_ID, count));
    }

    public List<Long> generatePhraseIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.PHID, count));
    }

    public List<Long> generateBsExportIterId(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.BSEXPORT_ITER_ID, count));
    }

    public List<Long> generatePerformanceFilterIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.PERF_FILTER_ID, count));
    }

    public List<Long> generateTurboAppInfoIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.TURBO_APP_INFO_ID, count));
    }

    public List<Long> generateDynamicIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.DYN_ID, count));
    }

    public List<Long> generateDynamicConditionIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.DYN_COND_ID, count));
    }

    public List<Long> generateCaesarIterId(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CAESAR_ITER_ID, count));
    }

    public List<Long> generateDspCreativeCaesarIterId(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.DSP_CREATIVE_CAESAR_ITER_ID, count));
    }

    public List<Long> generateBsOrderIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.BS_ORDER_ID, count));
    }

    public List<Long> generateRetargetingConditionIds(List<Long> clientIds) {
        return longValues(
                shardSupport.generateValues(ShardKey.RET_COND_ID, ShardKey.CLIENT_ID, clientIds));
    }

    public List<Long> generateRetargetingIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.RET_ID, count));
    }

    public List<Long> generateMinusWordsIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.MW_ID, count));
    }

    public List<Long> generateVcardIds(long clientId, int count) {
        return longValues(shardSupport.generateValues(ShardKey.VCARD_ID, ShardKey.CLIENT_ID, repeat(clientId, count)));
    }

    public List<Long> generatePromoExtensionIds(long clientId, int count) {
        return longValues(shardSupport.generateValues(ShardKey.PROMOACTION_ID, ShardKey.CLIENT_ID, repeat(clientId, count)));
    }

    public List<Long> generateTagIds(long clientId, int count) {
        return longValues(shardSupport.generateValues(ShardKey.TAG_ID, ShardKey.CLIENT_ID, repeat(clientId, count)));
    }

    public List<Long> generateAddressIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.AID, count));
    }

    public List<Long> generateOrgDetailsIds(long clientId, int count) {
        return longValues(shardSupport.generateValues(ShardKey.ORG_DETAILS_ID, ShardKey.CLIENT_ID,
                repeat(clientId, count)));
    }

    public List<Long> generateMapsIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.MAPS_ID, count));
    }

    public List<Long> generateBannerCreativeIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.BANNER_CREATIVE_ID, count));
    }

    // используется только в тестах
    public List<Long> generateOptCampRequestIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.OPT_CAMP_REQUEST_ID, count));
    }

    public List<Long> generateMobileContentIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.MOBILE_CONTENT_ID, count));
    }

    public List<Long> generateContentPromotionIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CONTENT_PROMOTION_ID, count));
    }

    // используется только в тестах
    public List<Long> generateClientDomainsRecordIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENT_DOMAINS_RECORD_ID, count));
    }

    public List<Long> generateFeedIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.FEED_ID, count));
    }

    public List<Long> generateFilteredFeedIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.FILTERED_FEED_ID, count));
    }

    public List<Long> generateImageIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.IMAGE_ID, count));
    }

    public List<Long> generateHierarchicalMultiplierIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.HIERARCHICAL_MULTIPLIER_ID, count));
    }

    public List<Long> generateDbQueueJobIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.JOB_ID, count));
    }

    public List<Long> generateDealNotificationIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.DEAL_NOTIFICATION_ID, count));
    }

    public List<Long> generateMdsFileIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.MDS_ID, count));
    }

    public List<Long> generateFreelancerProjectIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.FREELANCER_PROJECT_ID, count));
    }

    public List<Long> generateClientRelationIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENTS_RELATION_ID, count));
    }

    public List<Long> generateClientAvatarIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENTS_AVATARS_ID, count));
    }

    public List<Long> generateFreelancersCardIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.FREELANCERS_CARD_ID, count));
    }

    public List<Long> generateMobileAppIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.MOBILE_APP_ID, count));
    }

    public List<Long> generateMobileAppGoalIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.MOBILE_APP_GOAL_ID, count));
    }

    public List<Long> generateMobileAppTrackerIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.MOBILE_APP_TRACKER_ID, count));
    }

    public List<Long> generatePixelIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.PIXEL_ID, count));
    }

    public List<Long> generateAgencyOfflineReportIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.AGENCY_OFFLINE_REPORT, count));
    }

    public List<Long> generateOfflineReportIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.OFFLINE_REPORT, count));
    }

    public List<Long> generateBannerPoolImageIds(long clientId, int size) {
        return longValues(shardSupport
                .generateValues(ShardKey.BANNER_IMAGES_POOL_ID, ShardKey.CLIENT_ID, repeat(clientId, size)));
    }

    public List<Long> generateAdGroupAdditionalTargetingIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.ADGROUP_ADDITIONAL_TARGETING_ID, count));
    }

    public List<Long> generateModerateBannerPageIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.MODERATE_BANNER_PAGE_ID, count));
    }

    public List<Long> generateClientDialogIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENT_DIALOG_ID, count));
    }

    public List<Long> generateClientPhoneIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENT_PHONE_ID, count));
    }

    public List<Long> generateCalltrackingSettingsId(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CALLTRACKING_SETTINGS_ID, count));
    }

    public List<Long> generateDaasBriefIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.DAAS_BRIEF_ID, count));
    }

    public List<Long> generateBannerAdditionalHrefIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.BANNER_ADDITIONAL_HREFS_ID, count));
    }

    public List<Long> generateClientsCashbackProgramsIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENTS_CASHBACK_PROGRAM_ID, count));
    }

    public List<Long> generateClientCashbacksHistoryIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENT_CASHBACK_HISTORY_ID, count));
    }

    public List<Long> generateClientCashbackDetailsIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENT_CASHBACK_DETAILS_ID, count));
    }

    public List<Long> generateCampAdditionalTargetingsIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CAMP_ADDITIONAL_TARGETINGS_ID, count));
    }

    public List<Long> generateClientAdditionalTargetingsIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENT_ADDITIONAL_TARGETINGS_ID, count));
    }

    public List<Long> generateBannerMulticardIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.BANNER_MULTICARDS_MULTICARD_ID, count));
    }

    public List<Long> generateCampaignsPromotionsIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CAMPAIGNS_PROMOTIONS_ID, count));
    }

    public List<Long> generateClientMccRequestIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.CLIENT_MCC_REQUEST_ID, count));
    }

    public List<Long> generateReverseClientsRelationsIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.REVERSE_CLIENTS_RELATIONS_ID, count));
    }

    public List<Long> generateLalSegmentIds(int count) {
        return longValues(shardSupport.generateValues(AutoIncrementKey.LAL_SEGMENTS_ID, count));
    }

    /**
     * Проверяет, присутствует ли указанный {@code clientId} в системе
     *
     * @param clientId clientId для проверки
     * @return {@code true}, если {@code clientId} присутствует
     */
    public boolean isExistentClientId(long clientId) {
        return getShardByClientId(ClientId.fromLong(clientId)) != NO_SHARD;
    }

    /**
     * Возвращает мапу clientId -> exists по списку clientId
     */
    public Map<Long, Boolean> getExistingClientIds(List<Long> clientIds) {
        return EntryStream.of(getShardsByClientIds(clientIds))
                .mapValues(Objects::nonNull)
                .toMap();
    }

    /**
     * Определить существующие ClientId
     *
     * @param clientIds        — проверяемый список ClientId
     * @param includeShardZero — считать ли клиентов в 0-м шарде существующими
     * @return список существующих ClientId
     */
    public List<ClientId> getExistingClientIdsList(List<ClientId> clientIds, boolean includeShardZero) {
        return EntryStream.of(getShardsByClientIds(clientIds.stream().map(ClientId::asLong).collect(toList())))
                .filterValues(Objects::nonNull).filterValues(i -> includeShardZero || i != 0)
                .keys().map(ClientId::fromLong).toList();
    }

    /**
     * Проверяет, присутствует ли указанный {@code uid} в системе
     *
     * @param uid uid для проверки
     * @return {@code true}, если {@code uid} присутствует
     */
    public boolean isExistentUid(Long uid) {
        return null != shardSupport.getValue(ShardKey.UID, uid, ShardKey.CLIENT_ID, Long.class);
    }

    /**
     * Выделить shard для нового клиента.
     * <p>
     * Если клиент с указанным {@code clientClientId} ранее существовал, но был удалён,
     * переиспользуем шард, в котором он был.
     *
     * @return Номер shard-а
     */
    public int allocShardForNewClient(
            Long clientUid, String clientLogin, Long clientClientId) {
        int existingShard = getShardByClientId(ClientId.fromLong(clientClientId));
        int selectedShard;
        if (existingShard != NO_SHARD) {
            // Скрипт очистки пустых клиентов ppcClearEmptyClients.pl оставляет информацию о шарде.
            // Если для клиента указан шард, используем его
            selectedShard = existingShard;
        } else {
            selectedShard = shardSupport.selectShardForNewClient();
            shardSupport.saveValue(
                    ShardKey.CLIENT_ID, clientClientId, ShardKey.SHARD, selectedShard);
        }

        shardSupport.saveValue(
                ShardKey.UID, clientUid, ShardKey.CLIENT_ID, clientClientId);
        shardSupport.saveValue(
                ShardKey.LOGIN, clientLogin, ShardKey.UID, clientUid);

        return selectedShard;
    }

    private static List<Long> longValues(Iterable<? extends Number> numbers) {
        return mapList(numbers, Number::longValue);
    }

    private static <T> List<T> repeat(T obj, int times) {
        return Stream.generate(() -> obj).limit(times).collect(toList());
    }
}
