package ru.yandex.direct.dbutil.sharding;

import java.util.Map;

import com.google.common.collect.ImmutableMap;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Table;

import ru.yandex.direct.dbschema.ppcdict.Tables;

/**
 * Keys supported by the sharding system
 */
public enum ShardKey {
    SHARD("shard", ShardKeyType.INT, true),

    CLIENT_ID(Tables.SHARD_CLIENT_ID, SHARD),
    UID(Tables.SHARD_UID, CLIENT_ID),
    LOGIN(Tables.SHARD_LOGIN, UID, ShardKeyType.STR),
    ORDER_ID(Tables.SHARD_ORDER_ID, CLIENT_ID),
    CREATIVE_ID(Tables.SHARD_CREATIVE_ID, CLIENT_ID),

    // --- sharded auto-increment keys ---

    CID(Tables.SHARD_INC_CID, CLIENT_ID, true),
    STRATEGY_ID(Tables.SHARD_INC_STRATEGY_ID, CLIENT_ID, true),
    BID(Tables.SHARD_INC_BID, CLIENT_ID, true),
    MBID(Tables.SHARD_INC_MBID, CLIENT_ID, true),
    PID(Tables.SHARD_INC_PID, CLIENT_ID, true),
    TAG_ID(Tables.SHARD_INC_TAG_ID, CLIENT_ID, true),
    SITELINKS_SET_ID(Tables.SHARD_INC_SITELINKS_SET_ID, CLIENT_ID, true),
    RET_COND_ID(Tables.SHARD_INC_RET_COND_ID, CLIENT_ID, true),
    ORG_DETAILS_ID(Tables.SHARD_INC_ORG_DETAILS_ID, CLIENT_ID, true),
    VCARD_ID(Tables.SHARD_INC_VCARD_ID, CLIENT_ID, true),
    PROMOACTION_ID(Tables.SHARD_INC_PROMOACTION_ID, CLIENT_ID, true),
    BANNER_IMAGES_POOL_ID(Tables.SHARD_INC_BANNER_IMAGES_POOL_ID, CLIENT_ID, true),
    MEDIAPLAN_BID(Tables.SHARD_INC_MEDIAPLAN_BID, CLIENT_ID, true);

    private final String name;
    private final ShardKeyType type;
    private final Table<? extends Record> table;
    private final Field<?> keyField;
    private final ShardKey chainKey;
    private final Field<?> valueField;
    private final boolean nonZero;
    private final boolean autoIncrement;

    ShardKey(String name, ShardKeyType type, boolean nonZero) {
        this.name = name;
        this.type = type;
        this.table = null;
        this.keyField = null;
        this.chainKey = null;
        this.valueField = null;
        this.nonZero = nonZero;
        this.autoIncrement = false;
    }

    ShardKey(Table<? extends Record> table, ShardKey chainKey) {
        this(table, chainKey, ShardKeyType.INT);
    }

    ShardKey(Table<? extends Record> table, ShardKey chainKey, ShardKeyType type) {
        this(table, chainKey, type, false);
    }

    ShardKey(Table<? extends Record> table, ShardKey chainKey, boolean autoIncrement) {
        this(table, chainKey, ShardKeyType.INT, autoIncrement);
    }

    ShardKey(Table<? extends Record> table, ShardKey chainKey, ShardKeyType type, boolean autoIncrement) {
        Field<?>[] fields = table.fields();
        if (fields.length != 2) {
            throw new IllegalArgumentException("Table " + table.getName() + " must have exactly 2 columns");
        }
        this.name = fields[0].getName();
        this.type = type;
        this.table = table;
        this.keyField = fields[0];
        this.chainKey = chainKey;
        this.valueField = fields[1];
        this.nonZero = false;
        this.autoIncrement = autoIncrement;
        if (!valueField.getName().equals(chainKey.getName())) {
            throw new IllegalArgumentException(
                    "Table " + table.getName() + " has value field " + valueField.getName() + " but chain key "
                            + chainKey.getName() + " is expected");
        }
    }

    public String getName() {
        return name;
    }

    public ShardKeyType getType() {
        return type;
    }

    public boolean isRoot() {
        return table == null;
    }

    public Table<? extends Record> getTable() {
        return table;
    }

    public Field<?> getKeyField() {
        return keyField;
    }

    public ShardKey getChainKey() {
        if (isRoot()) {
            throw new IllegalStateException("Key " + getName() + " is the root key");
        }
        return chainKey;
    }

    public Field<?> getValueField() {
        if (isRoot()) {
            throw new IllegalStateException("Key " + getName() + " is the root key");
        }
        return valueField;
    }

    public boolean isNonZero() {
        return nonZero;
    }

    public boolean isAutoIncrement() {
        return autoIncrement;
    }

    private static final Map<String, ShardKey> keyByName;

    static {
        ImmutableMap.Builder<String, ShardKey> tmp = ImmutableMap.builder();
        for (ShardKey key : ShardKey.values()) {
            tmp.put(key.getName(), key);
        }
        keyByName = tmp.build();
    }

    public static ShardKey byName(String name) {
        ShardKey key = keyByName.get(name);
        if (key == null) {
            throw new IllegalArgumentException("Unknown shard key " + name);
        }
        return key;
    }
}
