package ru.yandex.direct.dbutil.sharding;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;

import com.google.common.collect.Lists;
import one.util.streamex.EntryStream;

import ru.yandex.direct.dbutil.SqlUtils;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.stream.Collectors.toList;

/**
 * Класс для разбиения и обработки списка объектов, находящихся в разных шардах
 *
 * @param <T> класс элемента списка
 */
public class ShardedData<T> {
    private Map<Integer, List<T>> shardedDataMap;
    private int chunkSize = 0;

    protected ShardedData(Map<Integer, List<T>> shardedDataMap) {
        this.shardedDataMap = checkNotNull(shardedDataMap, "shardedDataMap cannot be null");
    }

    /**
     * выполнение consumer для всех объектов пошардово
     *
     * @param consumer в параметрах принимает шард и список объектов, находящихся в этом шарде
     */
    public void forEach(BiConsumer<Integer, List<T>> consumer) {
        stream().forEach(e -> consumer.accept(e.getKey(), e.getValue()));
    }

    /**
     * При установке значения объекты будут обрабатываться чанками {@link SqlUtils#TYPICAL_SELECT_CHUNK_SIZE} size
     * chunkSize = 0: не разбивать на чанки
     */
    public ShardedData<T> chunkedByDefault() {
        return chunkedBy(SqlUtils.TYPICAL_SELECT_CHUNK_SIZE);
    }

    /**
     * При установке значения объекты будут обрабатываться чанками установленного size
     * chunkSize = 0: не разбивать на чанки
     */
    public ShardedData<T> chunkedBy(int chunkSize) {
        this.chunkSize = chunkSize;
        return this;
    }

    /**
     * ключ = шард, значение = список элементов из data, которые находятся в данном шарде
     */
    public Map<Integer, List<T>> getShardedDataMap() {
        return shardedDataMap;
    }

    /**
     * Получение {@link EntryStream} для списка объектов, разделенных на чанки
     */
    public EntryStream<Integer, List<T>> stream() {
        return EntryStream.of(shardedDataMap)
                .sortedByInt(Map.Entry::getKey)
                .flatMapValues(x -> splitByChunks(x).stream());
    }

    /**
     * Получить список элементов, находящихся во всех шардах
     */
    public List<T> getData() {
        return shardedDataMap.values()
                .stream()
                .flatMap(List::stream)
                .collect(toList());
    }

    private List<List<T>> splitByChunks(List<T> list) {
        if (chunkSize == 0) {
            return Collections.singletonList(list);
        }
        return Lists.partition(list, chunkSize);
    }
}
