package ru.yandex.direct.dbutil.wrapper;

import java.sql.SQLException;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;

import javax.sql.DataSource;

import com.google.common.primitives.Ints;
import com.mysql.cj.jdbc.MysqlDataSource;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.clickhouse.ClickHouseDataSource;
import ru.yandex.clickhouse.settings.ClickHouseProperties;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.db.config.DbConfig;

import static com.google.common.base.Strings.isNullOrEmpty;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.DateTimeUtils.MOSCOW_TIMEZONE;

@Component
public class DataSourceFactory {
    private static final Logger logger = LoggerFactory.getLogger(DataSourceFactory.class);

    private static final String ZERO_DATETIME_BEHAVIOR_CONVERT_TO_NULL = "CONVERT_TO_NULL";
    private static final AtomicLong SERIAL = new AtomicLong();
    private final DirectPoolConfig poolConfig;
    private final DirectConfig clickhouseOptions;
    private final Map<String, String> dbUsers;

    public DataSourceFactory(DirectConfig directConfig) {
        poolConfig = DirectPoolConfig.from(directConfig.getBranch("db_pool"));
        clickhouseOptions = directConfig.getBranch("clickhouse_options");
        dbUsers = directConfig.findBranch("db_user").map(DirectConfig::asMap)
                .orElse(Map.of());

        logger.info("poolConfig: {}", poolConfig);
    }

    private static ClickHouseProperties clickhouseDefaults() {
        ClickHouseProperties props = new ClickHouseProperties();
        props.setMaxExecutionTime(Ints.saturatedCast(Duration.ofSeconds(300).getSeconds()));
        props.setSocketTimeout(Ints.saturatedCast(Duration.ofSeconds(300).toMillis()));
        props.setConnectionTimeout(Ints.saturatedCast(Duration.ofSeconds(4).toMillis()));
        props.setKeepAliveTimeout(Ints.saturatedCast(Duration.ofSeconds(2).toMillis()));
        props.setTimeToLiveMillis(Ints.saturatedCast(Duration.ofSeconds(2).toMillis()));
        return props;
    }

    private DataSource createSimpleDataSource(DbConfig config) {
        DbConfig.Engine engine = config.getEngine();
        if (engine == DbConfig.Engine.CLICKHOUSE) {
            return new ClickHouseDataSource(
                    config.getJdbcUrl(),
                    makeClickHouseProperties(config)
            );
        } else if (engine == DbConfig.Engine.MYSQL) {
            MysqlDataSource mysqlDataSource = new MysqlDataSource();
            mysqlDataSource.setUrl(getMySqlJdbcUrl(config));
            addAuth(config, (user, pass) -> {
                mysqlDataSource.setUser(user);
                mysqlDataSource.setPassword(pass);
            });
            try {
                // Делаем проверку SQL-запросов в mysql строже(и ближе к 'стандартному' поведению).
                // Такой sql_mode станет режимом по-умолчанию в mysql 5.7(но будет отдельно настраиваться в сторону
                // ослабления, поскольку в Perl-директе есть несовместимые запросы)
                mysqlDataSource.setSessionVariables("range_optimizer_max_mem_size=134217728;"
                        + "sql_mode="
                        + "'ONLY_FULL_GROUP_BY,"
                        + "STRICT_TRANS_TABLES,"
                        + "NO_ZERO_IN_DATE,"
                        // + "NO_ZERO_DATE,"      // закомментировали, так как в базе
                        // используются даты 0000-00-00 00:00:00
                        + "ERROR_FOR_DIVISION_BY_ZERO,"
                        + "NO_AUTO_CREATE_USER,"
                        + "NO_ENGINE_SUBSTITUTION'");
                // для чтения из базы дат вида 0000-00-00 00:00:00
                mysqlDataSource.setZeroDateTimeBehavior(ZERO_DATETIME_BEHAVIOR_CONVERT_TO_NULL);
                mysqlDataSource.setServerTimezone(MOSCOW_TIMEZONE);
            } catch (SQLException e) {
                throw new IllegalArgumentException(e);
            }
            return mysqlDataSource;
        } else {
            throw new UnsupportedDbEngineException("Unsupported engine in db-config: '" + engine + "'");
        }
    }

    private ClickHouseProperties makeClickHouseProperties(DbConfig config) {
        ClickHouseProperties props = clickhouseDefaults();
        addAuth(config, (user, pass) -> {
            if (!isNullOrEmpty(user)) {
                props.setUser(user);
            }
            if (!isNullOrEmpty(pass)) {
                props.setPassword(pass);
            }
        });

        props.setSsl(config.isSsl());
        if (config.isSsl()) {
            props.setSslMode(config.isVerify() ? "strict" : "none");
        }
        if (config.getClickhouseMaxBlockSize() != null) {
            props.setMaxBlockSize(config.getClickhouseMaxBlockSize());
        }
        clickhouseOptions.findLong("max_memory_usage")
                .ifPresent(props::setMaxMemoryUsage);
        clickhouseOptions.findLong("max_memory_usage_for_user")
                .ifPresent(props::setMaxMemoryUsageForUser);
        clickhouseOptions.findLong("max_memory_usage_for_all_queries")
                .ifPresent(props::setMaxMemoryUsageForAllQueries);
        return props;
    }

    /**
     * Для ppcdict хардкодим опцию {@code allowMultiQueries=true}, чтобы не зависеть от конфигов.
     * Опция критичная, позволяет ускорить массовую генерацию id (см. ShardedValuesGenerator).
     */
    private static String getMySqlJdbcUrl(DbConfig config) {
        final String url = config.getJdbcUrl();
        if (config.getDbName().equalsIgnoreCase(SimpleDb.PPCDICT.toString())) {
            if (!url.contains("allowMultiQueries")) {
                return url + (url.indexOf('?') >= 0 ? '&' : '?') + "allowMultiQueries=true";
            }
        }
        return url;
    }

    public DataSource createDataSource(DbConfig config) {
        DataSource dataSource = createSimpleDataSource(config);
        if (!isPoolledDb(config)) {
            // Clickhouse не очень предназначен для использования с connection-pool:
            // он рвёт соединение через 3 сек, что меньше минимального validationTimeout
            return dataSource;
        }

        HikariConfig hikariConfig = new HikariConfig();
        hikariConfig.setPoolName(getPoolName(config));
        hikariConfig.setMaximumPoolSize(poolConfig.maxConnections);
        hikariConfig.setMinimumIdle(poolConfig.minIdleConnections);
        hikariConfig.setLeakDetectionThreshold(poolConfig.leakDetectionThreshold);
        hikariConfig.setInitializationFailTimeout(-1);
        hikariConfig.setConnectionTimeout(config.getConnectTimeout(TimeUnit.MILLISECONDS));
        hikariConfig.setValidationTimeout(config.getConnectTimeout(TimeUnit.MILLISECONDS) / 2);
        hikariConfig.setDataSource(dataSource);
        hikariConfig.setRegisterMbeans(true);

        return new HikariDataSource(hikariConfig);
    }

    private static boolean isPoolledDb(DbConfig config) {
        return config.getEngine() != DbConfig.Engine.CLICKHOUSE;
    }

    private String getPoolName(DbConfig config) {
        // для того, чтобы при пересоздании пула имя было уникальное - добавляем в конец уникальный номер
        return config.getDbName().replace(':', '_') + "__" + SERIAL.incrementAndGet();
    }

    DirectPoolConfig getPoolConfig() {
        return poolConfig;
    }

    private void addAuth(DbConfig config, BiConsumer<String, String> consumer) {
        var extraUser = nullableNvl(
                dbUsers.get(config.getDbName()),
                dbUsers.get("default")
        );
        if (extraUser != null) {
            consumer.accept(extraUser, config.getExtraUsers().get(extraUser));
        } else {
            consumer.accept(config.getUser(), config.getPass());
        }
    }

    static class DirectPoolConfig {
        static final int DEFAULT_MAX_CONNECTIONS = 20;
        static final int DEFAULT_MIN_IDLE_CONNECTIONS = 5;
        static final long DEFAULT_LEAK_DETECTION_THRESHOLD = 0;

        final int maxConnections;
        final int minIdleConnections;
        final long leakDetectionThreshold;

        private DirectPoolConfig(int maxConnections, int minIdleConnections, long leakDetectionThreshold) {
            this.maxConnections = maxConnections;
            this.minIdleConnections = minIdleConnections;
            this.leakDetectionThreshold = leakDetectionThreshold;
        }

        public static DirectPoolConfig from(DirectConfig cfg) {
            Long leakDetectionThreshold = cfg.findDuration("leak_detection_threshold")
                    .map(Duration::toMillis)
                    .orElse(DEFAULT_LEAK_DETECTION_THRESHOLD);

            return new DirectPoolConfig(
                    cfg.findInt("max_connections").orElse(DEFAULT_MAX_CONNECTIONS),
                    cfg.findInt("min_idle_connections").orElse(DEFAULT_MIN_IDLE_CONNECTIONS),
                    leakDetectionThreshold
            );
        }

        @Override
        public String toString() {
            return "DirectPoolConfig{" +
                    "maxConnections=" + maxConnections +
                    ", minIdleConnections=" + minIdleConnections +
                    ", leakDetectionThreshold=" + leakDetectionThreshold +
                    '}';
        }
    }
}
