package ru.yandex.direct.dbutil.wrapper;

import java.util.Collection;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.management.ObjectName;

import com.google.common.collect.ImmutableMap;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.graphite.GraphiteMetricsBuffer;
import ru.yandex.direct.metric.collector.MetricProvider;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.monlib.metrics.labels.Labels;

import static ru.yandex.direct.graphite.GraphiteUtils.normalizeMetricPart;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Сбор информации о hikari-пулах приложения
 */
@Component
@Lazy(false)
public class DatabasePoolMetricProvider implements MetricProvider {
    private static final String JMX_NAME_PATTERN = "com.zaxxer.hikari:type=Pool *";
    // регулярка для получения имени базы из имени jmx-bean-а
    // после двух подчёркиваний может быть произвольный рандомизирующий суффикс
    private static final Pattern NAME_PATTERN = Pattern.compile(".*Pool \\((.*?)(__.*)?\\)");

    private final DataSourceFactory dataSourceFactory;

    private static final ImmutableMap<String, String> ATTRS = new ImmutableMap.Builder<String, String>()
            .put("ActiveConnections", "active_conns")
            .put("IdleConnections", "idle_conns")
            .put("ThreadsAwaitingConnection", "waiting_threads")
            .put("TotalConnections", "total_conns")
            .build();

    public DatabasePoolMetricProvider(DataSourceFactory dataSourceFactory) {
        this.dataSourceFactory = dataSourceFactory;
    }

    @Override
    public void provideMetrics(GraphiteMetricsBuffer unused) {
        Collection<ObjectName> objNames = SolomonUtils.findJmxBeans(JMX_NAME_PATTERN);
        for (ObjectName objName : objNames) {
            Labels commonLabels = Labels.of(
                    "type", "pool",
                    "pool_name", normalizeMetricPart(getPoolName(objName))
            );
            SolomonUtils.registerJmxAttributes("db", commonLabels, objName, ATTRS);

            SOLOMON_REGISTRY.lazyGaugeInt64(
                    "max_conns",
                    commonLabels,
                    () -> dataSourceFactory.getPoolConfig().maxConnections);
        }
    }

    private String getPoolName(ObjectName objName) {
        Matcher matcher = NAME_PATTERN.matcher(objName.getCanonicalName());
        if (!matcher.find()) {
            throw new IllegalStateException("Can't find pool name from '" + objName.getCanonicalName() + "'");
        }
        return matcher.group(1);
    }
}
