package ru.yandex.direct.dbutil.wrapper;

import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.concurrent.atomic.AtomicReference;

import javax.sql.DataSource;

/**
 * This class implements DataSource interface and delegates
 * all it's methods to wrapped DataSource object.
 * <p>
 * Wrapped DataSource object can be replaced by another at any time.
 */
class RefreshableDataSource implements DataSource, AutoCloseable {

    private AtomicReference<DataSource> dataSourceReference;

    public RefreshableDataSource(DataSource dataSource) {
        this.dataSourceReference = new AtomicReference<>(dataSource);
    }

    public DataSource refresh(DataSource dataSource) {
        return dataSourceReference.getAndSet(dataSource);
    }

    public DataSource getDataSource() {
        return dataSourceReference.get();
    }

    // DataSource wrapped methods

    @Override
    public Connection getConnection() throws SQLException {
        return getDataSource().getConnection();
    }

    @Override
    public Connection getConnection(String username, String password) throws SQLException {
        return getDataSource().getConnection(username, password);
    }

    @Override
    public PrintWriter getLogWriter() throws SQLException {
        return getDataSource().getLogWriter();
    }

    @Override
    public void setLogWriter(PrintWriter out) throws SQLException {
        getDataSource().setLogWriter(out);
    }

    @Override
    public void setLoginTimeout(int seconds) throws SQLException {
        getDataSource().setLoginTimeout(seconds);
    }

    @Override
    public int getLoginTimeout() throws SQLException {
        return getDataSource().getLoginTimeout();
    }

    @Override
    public java.util.logging.Logger getParentLogger() throws SQLFeatureNotSupportedException {
        return getDataSource().getParentLogger();
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        return getDataSource().unwrap(iface);
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return getDataSource().isWrapperFor(iface);
    }

    @Override
    public void close() throws Exception {
        if (getDataSource() instanceof AutoCloseable) {
            ((AutoCloseable) getDataSource()).close();
        }
    }
}
