package ru.yandex.direct.dbutil.wrapper;


import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.StringJoiner;

import com.google.common.collect.ImmutableSet;
import org.jooq.ExecuteContext;
import org.jooq.impl.DefaultExecuteListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Слушатель, выводящий stacktrace для каждого запроса (этого не хватает в org.jooq.tools.LoggerListener)
 * (предполагается, что запрос и выведет оригинальный логгер)
 */
public class StackTraceLoggerListener extends DefaultExecuteListener {
    // выводятся фреймы только с таким префиксом
    // jooq.* не выводим чтобы не загоромождать лог
    private static final String PACKAGE_PREFIX = "ru.yandex.direct.";
    private static final String SKIP_PATTERN = "EnhancerBySpringCGLIB";
    private static final Set<String> STOP_CLASSES = ImmutableSet.of(
            "javax.servlet.http.HttpServlet"
    );

    private static final Logger logger = LoggerFactory.getLogger(StackTraceLoggerListener.class);
    public static final String SELF_NAME = StackTraceLoggerListener.class.getCanonicalName();

    private String dbname;

    public StackTraceLoggerListener(String dbname) {
        this.dbname = dbname;
    }

    @Override
    public void end(ExecuteContext ctx) {
        if (logger.isDebugEnabled() && ctx.batchSQL().length > 0) {
            StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
            logger.debug("Query for {}\n{}", dbname, formatStackTrace(stackTrace));
        }
    }

    private String formatStackTrace(StackTraceElement[] stackTrace) {
        StringJoiner joiner = new StringJoiner("\n");
        List<StackTraceElement> skipped = new ArrayList<>();

        boolean stop = false;
        for (StackTraceElement el : stackTrace) {
            String className = el.getClassName();
            if (STOP_CLASSES.contains(className)) {
                stop = true;
            }
            if (!stop
                    && className.startsWith(PACKAGE_PREFIX)
                    && !className.equals(SELF_NAME)
                    && !className.contains(SKIP_PATTERN)
            ) {
                addSkippedAndClear(joiner, skipped);
                joiner.add("\tat " + el);
            } else {
                skipped.add(el);
            }
        }

        addSkippedAndClear(joiner, skipped);

        return joiner.toString();
    }

    private void addSkippedAndClear(StringJoiner sj, List<StackTraceElement> skipped) {
        if (skipped.size() == 1) {
            // если нужно пропустить только один фрейм - проще его вывести, место не теряется
            sj.add("\tat " + skipped.get(0));
            skipped.clear();
        } else if (skipped.size() > 1) {
            sj.add("\t... " + skipped.size() + " stack frames skipped");
            skipped.clear();
        }
    }
}
