package ru.yandex.direct.ess.client

import com.fasterxml.jackson.databind.JavaType
import org.apache.commons.lang3.builder.ReflectionToStringBuilder
import org.reflections.Reflections
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.ess.client.repository.EssAdditionalObjectsRepository
import ru.yandex.direct.ess.common.models.BaseEssConfig
import ru.yandex.direct.ess.common.models.BaseLogicObject
import ru.yandex.direct.utils.JsonUtils
import java.lang.reflect.Modifier
import java.util.List.copyOf

@Component
class EssClient(private val essAdditionalObjectsRepository: EssAdditionalObjectsRepository, packagePath: String) {
    private val logicProcessorToObjectClass: Map<String, BaseEssConfig>
    private val essConfigsClasses: List<Class<out BaseEssConfig>>

    companion object {
        private const val DEFAULT_PACKAGE = "ru.yandex.direct.ess.config"
        private const val INSERT_CHUNK_SIZE = 1_000
    }

    init {
        essConfigsClasses = calculateEssConfigClasses(packagePath)
        logicProcessorToObjectClass = logicProcessorToObjectClassMap
    }

    val logicProcessorsNames: Collection<String>
        get() = logicProcessorToObjectClass.keys

    @Autowired
    constructor(essAdditionalObjectsRepository: EssAdditionalObjectsRepository) : this(
        essAdditionalObjectsRepository,
        DEFAULT_PACKAGE
    )

    private fun calculateEssConfigClasses(packagePath: String) =
        Reflections(packagePath).getSubTypesOf(BaseEssConfig::class.java)
            .filter { clazz -> !clazz.isInterface && !Modifier.isAbstract(clazz.modifiers) }

    fun getTopicForProcessor(processorName: String) =
        logicProcessorToObjectClass[processorName]?.topic
            ?: throw IllegalStateException("Unexpected processor name $processorName")

    fun getLogicObjectTypeForProcessor(logicProcessorName: String) =
        logicProcessorToObjectClass[logicProcessorName]?.logicObject
            ?: throw IllegalStateException("Unsupported logic processor $logicProcessorName")

    fun addLogicObjectsForProcessor(
        shard: Int,
        logicProcessorName: String,
        logicObjects: List<BaseLogicObject>,
        withDelete: Boolean = true
    ) {
        logicObjects
            .map {
                try {
                    JsonUtils.toJson(it)
                } catch (e: RuntimeException) {
                    throw IllegalStateException("Can't serialize logic object " + ReflectionToStringBuilder.toString(it))
                }
            }
            .chunked(INSERT_CHUNK_SIZE)
            .forEach {
                essAdditionalObjectsRepository.addLogicObjectsForProcessor(shard, logicProcessorName, it)
                if (withDelete) essAdditionalObjectsRepository.clearLogicObjects(shard, it.size)
            }
    }

    fun addLogicObjectsForProcessor(
        shard: Int,
        logicProcessorName: String,
        logicObjects: String,
        withDelete: Boolean = true
    ) {
        val logicObjectClass = getLogicObjectTypeForProcessor(logicProcessorName)

        val logicObjectsDeserialized: List<BaseLogicObject> = try {
            val listLogicObjectsType: JavaType = JsonUtils.getTypeFactory().constructCollectionType(
                MutableList::class.java,
                logicObjectClass
            )
            JsonUtils.fromJson(logicObjects, listLogicObjectsType)
        } catch (e: RuntimeException) {
            throw IllegalStateException("Can't deserialize logic objects", e)
        }

        addLogicObjectsForProcessor(shard, logicProcessorName, logicObjectsDeserialized, withDelete)
    }

    fun clearLogicObjects(shard: Int, limit: Int? = null) =
        essAdditionalObjectsRepository.clearLogicObjects(shard, limit)

    private val logicProcessorToObjectClassMap: Map<String, BaseEssConfig>
        get() {
            return essConfigsClasses.map {
                try {
                    it.getDeclaredConstructor().newInstance()
                } catch (e: RuntimeException) {
                    throw IllegalStateException("Failed to init ess config for class " + it.simpleName, e)
                }
            }.associateBy { it.logicProcessName }
        }

    fun getEssConfigsClasses(): List<Class<out BaseEssConfig>> = copyOf(essConfigsClasses)
}
