package ru.yandex.direct.excel.processing.model.internalad;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingJoinType;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdditionalTargetingValue;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.HasLCookieAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.HasPassportIdAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.InterfaceLang;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.InterfaceLangsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.InternalNetworkAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsDefaultYandexSearchAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsPPLoggedInAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsVirusedAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.IsYandexPlusAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.MobileInstalledApp;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.MobileInstalledAppsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.IsMobileAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.IsTabletAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.IsTouchAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.UatraitsTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.uatraits.model.VersionedTargeting;
import ru.yandex.direct.excel.processing.model.VersionedTargetingInfo;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.excel.processing.model.internalad.TargetingStore.getDefaultJoinTypeByTargetingMode;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@ParametersAreNonnullByDefault
public class AdGroupAdditionalTargetingRepresentation {

    private final TargetingStore targetingStore;

    public List<AdGroupAdditionalTargeting> getTargetingList() {
        return List.copyOf(targetingStore.getTargetings());
    }

    /**
     * Возвращает значение таргетинга прочитанного из excel файла
     */
    public Object getTargetingOriginalValueFromExcel(AdGroupAdditionalTargeting additionalTargeting) {
        return targetingStore.getTargetingOriginalValueFromExcel(additionalTargeting);
    }

    public AdGroupAdditionalTargetingRepresentation() {
        this.targetingStore = new TargetingStore();
    }

    public AdGroupAdditionalTargetingRepresentation(Collection<? extends AdGroupAdditionalTargeting> additionalTargetings) {
        this.targetingStore = new TargetingStore(additionalTargetings);
    }

    @Nullable
    public Boolean getHasPassportIdAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(HasPassportIdAdGroupAdditionalTargeting.class);
    }

    public void setHasPassportIdAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, HasPassportIdAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public Boolean getIsVirusedAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(IsVirusedAdGroupAdditionalTargeting.class);
    }

    public void setIsVirusedAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, IsVirusedAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public Boolean getIsYandexPlusAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(IsYandexPlusAdGroupAdditionalTargeting.class);
    }

    public void setIsYandexPlusAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, IsYandexPlusAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public Boolean getHasLCookieAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(HasLCookieAdGroupAdditionalTargeting.class);
    }

    public void setHasLCookieAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, HasLCookieAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public Boolean getInternalNetworkAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(InternalNetworkAdGroupAdditionalTargeting.class);
    }

    public void setInternalNetworkAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, InternalNetworkAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public Boolean getIsPPLoggedInAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(IsPPLoggedInAdGroupAdditionalTargeting.class);
    }

    public void setIsPPLoggedInAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, IsPPLoggedInAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public Boolean getIsMobileAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(IsMobileAdGroupAdditionalTargeting.class);
    }

    public void setIsMobileAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, IsMobileAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public Boolean getIsTabletAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(IsTabletAdGroupAdditionalTargeting.class);
    }

    public void setIsTabletAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, IsTabletAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public Boolean getIsTouchAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(IsTouchAdGroupAdditionalTargeting.class);
    }

    public void setIsTouchAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, IsTouchAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public Set<InterfaceLang> getInterfaceLangsAdGroupAdditionalTargetingTargeting() {
        InterfaceLangsAdGroupAdditionalTargeting targeting =
                targetingStore.getAdGroupAdditionalTargeting(InterfaceLangsAdGroupAdditionalTargeting.class, true);
        return ifNotNull(targeting, InterfaceLangsAdGroupAdditionalTargeting::getValue);
    }

    public void setInterfaceLangsAdGroupAdditionalTargetingTargeting(@Nullable Set<InterfaceLang> values) {
        if (CollectionUtils.isNotEmpty(values)) {
            targetingStore.addAdditionalTargetingWithValue(InterfaceLangsAdGroupAdditionalTargeting::new,
                    InterfaceLangsAdGroupAdditionalTargeting.VALUE, true, values,
                    mapSet(values, InterfaceLang::getTypedValue));
        }
    }

    @Nullable
    public Set<InterfaceLang> getInterfaceLangsAdGroupAdditionalTargetingFiltering() {
        InterfaceLangsAdGroupAdditionalTargeting filtering =
                targetingStore.getAdGroupAdditionalTargeting(InterfaceLangsAdGroupAdditionalTargeting.class, false);
        return ifNotNull(filtering, InterfaceLangsAdGroupAdditionalTargeting::getValue);
    }

    public void setInterfaceLangsAdGroupAdditionalTargetingFiltering(@Nullable Set<InterfaceLang> values) {
        if (CollectionUtils.isNotEmpty(values)) {
            targetingStore.addAdditionalTargetingWithValue(InterfaceLangsAdGroupAdditionalTargeting::new,
                    InterfaceLangsAdGroupAdditionalTargeting.VALUE, false, values,
                    mapSet(values, InterfaceLang::getTypedValue));
        }
    }

    @Nullable
    public Boolean getIsDefaultYandexSearchAdGroupAdditionalTargeting() {
        return targetingStore.getBooleanAdditionalTargeting(IsDefaultYandexSearchAdGroupAdditionalTargeting.class);
    }

    public void setIsDefaultYandexSearchAdGroupAdditionalTargeting(@Nullable Boolean targeting) {
        targetingStore.addBooleanAdditionalTargeting(targeting, IsDefaultYandexSearchAdGroupAdditionalTargeting::new);
    }

    @Nullable
    public <T extends AdGroupAdditionalTargeting, P> P getAdditionalTargetingValue(
            ModelProperty<T, AdditionalTargetingValue<P>> valueModelProperty, boolean isTargeting) {
        T targeting = targetingStore.getAdGroupAdditionalTargeting(valueModelProperty.getModelClass(), isTargeting);
        return ifNotNull(ifNotNull(targeting, valueModelProperty::get), AdditionalTargetingValue::getValue);
    }

    public <T extends AdGroupAdditionalTargeting, P> void setAdditionalTargetingValue(
            Supplier<T> constructor,
            ModelProperty<T, AdditionalTargetingValue<P>> valueModelProperty,
            boolean isTargeting,
            @Nullable P value) {
        if (value != null) {
            targetingStore.addAdditionalTargetingWithValue(constructor, valueModelProperty, isTargeting,
                    AdditionalTargetingValue.of(value), value);
        }
    }

    @Nullable
    public <T extends AdGroupAdditionalTargeting, K, C extends Collection<K>> C getAdditionalTargetingWithCollectionValue(
            ModelProperty<T, C> valueModelProperty, boolean isTargeting) {
        T targeting = targetingStore.getAdGroupAdditionalTargeting(valueModelProperty.getModelClass(), isTargeting);
        return ifNotNull(targeting, valueModelProperty::get);
    }

    public <T extends AdGroupAdditionalTargeting, K, C extends Collection<K>> void setAdditionalTargetingWithCollectionValue(
            Supplier<T> constructor,
            ModelProperty<T, C> valueModelProperty,
            boolean isTargeting,
            @Nullable C values) {
        if (CollectionUtils.isNotEmpty(values)) {
            targetingStore
                    .addAdditionalTargetingWithValue(constructor, valueModelProperty, isTargeting, values, values);
        }
    }

    @Nullable
    public <T extends AdGroupAdditionalTargeting, V extends UatraitsTargeting> List<String> getAdditionalUatraitsTargeting(
            ModelProperty<T, List<V>> valueModelProperty,
            Map<Long, String> targetingValueEntryIdMapper,
            boolean isTargeting) {
        T targeting = targetingStore.getAdGroupAdditionalTargeting(valueModelProperty.getModelClass(), isTargeting);
        return ifNotNull(targeting, t -> {
            List<V> uatraitsTargetings = valueModelProperty.get(t);

            checkState(uatraitsTargetings != null, "values of uatraitsTargetings must not be null");
            return StreamEx.of(uatraitsTargetings)
                    .map(UatraitsTargeting::getTargetingValueEntryId)
                    .peek(id -> checkState(targetingValueEntryIdMapper.containsKey(id),
                            "not found value for targetingValueEntryId = %s", id))
                    .map(targetingValueEntryIdMapper::get)
                    .toList();
        });
    }

    public <T extends AdGroupAdditionalTargeting, V extends UatraitsTargeting> void setAdditionalUatraitsTargeting(
            Supplier<T> targetingConstructor, Supplier<V> valueConstructor,
            ModelProperty<T, List<V>> valueModelProperty,
            Map<String, Long> targetingValueEntryIdMapper,
            boolean isTargeting,
            @Nullable List<String> valuesFromExcel) {
        List<V> values = filterAndMapList(valuesFromExcel, StringUtils::isNotBlank, t -> {
            V v = valueConstructor.get();
            // приводим значение в нижний регистр, чтобы можно было получить entryId без учета регистра
            v.setTargetingValueEntryId(targetingValueEntryIdMapper.get(t.toLowerCase()));
            return v;
        });
        if (CollectionUtils.isNotEmpty(values)) {
            targetingStore.addAdditionalTargetingWithValue(targetingConstructor, valueModelProperty, isTargeting,
                    values, valuesFromExcel);
        }
    }

    @Nullable
    public <T extends AdGroupAdditionalTargeting> Set<String> getAdditionalDesktopTargeting(
            ModelProperty<T, Set<Long>> valueModelProperty,
            Map<Long, String> targetingValueEntryIdMapper,
            boolean isTargeting) {
        AdGroupAdditionalTargetingJoinType joinType = getDefaultJoinTypeByTargetingMode(isTargeting);
        return getAdditionalDesktopTargeting(valueModelProperty, targetingValueEntryIdMapper, isTargeting, joinType);
    }

    @Nullable
    public <T extends AdGroupAdditionalTargeting> Set<String> getAdditionalDesktopTargeting(
            ModelProperty<T, Set<Long>> valueModelProperty,
            Map<Long, String> targetingValueEntryIdMapper,
            boolean isTargeting,
            AdGroupAdditionalTargetingJoinType joinType) {
        T targeting = targetingStore.getAdGroupAdditionalTargeting(valueModelProperty.getModelClass(),
                isTargeting, joinType);
        return ifNotNull(targeting, t -> {
            Set<Long> targetings = valueModelProperty.get(t);
            checkState(targetings != null, "values of DesktopTargeting must not be null");
            return StreamEx.of(targetings)
                    .peek(value -> checkState(targetingValueEntryIdMapper.containsKey(value),
                            "not found value id = %s", value))
                    .map(targetingValueEntryIdMapper::get)
                    .toSet();
        });
    }

    public <T extends AdGroupAdditionalTargeting> void setAdditionalDesktopTargeting(
            Supplier<T> targetingConstructor,
            ModelProperty<T, Set<Long>> valueModelProperty,
            Map<String, Long> targetingValueEntryIdMapper,
            boolean isTargeting,
            @Nullable Set<String> valuesFromExcel) {
        AdGroupAdditionalTargetingJoinType joinType = getDefaultJoinTypeByTargetingMode(isTargeting);
        setAdditionalDesktopTargeting(targetingConstructor, valueModelProperty, targetingValueEntryIdMapper,
                isTargeting, joinType, valuesFromExcel);
    }

    public <T extends AdGroupAdditionalTargeting> void setAdditionalDesktopTargeting(
            Supplier<T> targetingConstructor,
            ModelProperty<T, Set<Long>> valueModelProperty,
            Map<String, Long> targetingValueEntryIdMapper,
            boolean isTargeting,
            AdGroupAdditionalTargetingJoinType joinType,
            @Nullable Set<String> valuesFromExcel) {
        Set<Long> values = filterAndMapToSet(valuesFromExcel, StringUtils::isNotBlank,
                // приводим значение в нижний регистр, чтобы можно было получить entryId без учета регистра
                t -> targetingValueEntryIdMapper.get(t.toLowerCase()));
        if (CollectionUtils.isNotEmpty(values)) {
            targetingStore.addAdditionalTargetingWithValue(targetingConstructor, valueModelProperty,
                    isTargeting, joinType, values, valuesFromExcel);
        }
    }

    @Nullable
    public Set<String> getMobileAppTargetingMapper(boolean isTargeting) {
        AdGroupAdditionalTargetingJoinType joinType = getDefaultJoinTypeByTargetingMode(isTargeting);
        return getMobileAppTargetingMapper(isTargeting, joinType);
    }

    @Nullable
    public Set<String> getMobileAppTargetingMapper(boolean isTargeting, AdGroupAdditionalTargetingJoinType joinType) {
        var targeting = targetingStore.getAdGroupAdditionalTargeting(
                MobileInstalledAppsAdGroupAdditionalTargeting.class, isTargeting, joinType);
        return ifNotNull(targeting, t -> {
            Set<MobileInstalledApp> targetings = MobileInstalledAppsAdGroupAdditionalTargeting.VALUE.get(t);
            checkState(targetings != null, "values of MobileAppTargeting must not be null");
            return StreamEx.of(targetings)
                    .map(MobileInstalledApp::getStoreUrl)
                    .toSet();
        });
    }

    public void setMobileAppTargetingMapper(boolean isTargeting, @Nullable Set<String> valuesFromExcel) {
        AdGroupAdditionalTargetingJoinType joinType = getDefaultJoinTypeByTargetingMode(isTargeting);
        setMobileAppTargetingMapper(isTargeting, joinType, valuesFromExcel);
    }

    public void setMobileAppTargetingMapper(boolean isTargeting, AdGroupAdditionalTargetingJoinType joinType,
                                            @Nullable Set<String> valuesFromExcel) {
        Set<MobileInstalledApp> values = filterAndMapToSet(valuesFromExcel, StringUtils::isNotBlank, t ->
                new MobileInstalledApp().withStoreUrl(t));
        if (CollectionUtils.isNotEmpty(values)) {
            targetingStore.addAdditionalTargetingWithValue(MobileInstalledAppsAdGroupAdditionalTargeting::new,
                    MobileInstalledAppsAdGroupAdditionalTargeting.VALUE, isTargeting, joinType, values,
                    valuesFromExcel);
        }
    }

    public <T extends AdGroupAdditionalTargeting, V extends VersionedTargeting>
    List<VersionedTargetingRepresentation> getVersionedAdditionalTargeting(ModelProperty<T, List<V>> valueModelProperty,
                                                                           Map<Long, String> targetingValueEntryIdMapper) {
        T positiveValue = targetingStore.getAdGroupAdditionalTargeting(valueModelProperty.getModelClass(), true);
        T negativeValue = targetingStore.getAdGroupAdditionalTargeting(valueModelProperty.getModelClass(), false);
        List<VersionedTargetingRepresentation> result = new ArrayList<>();
        if (positiveValue != null) {
            List<V> versionedTargetings = valueModelProperty.get(positiveValue);
            checkState(versionedTargetings != null, "values of versionedTargeting must not be null");
            result.addAll(mapList(versionedTargetings,
                    vt -> toVersionedTargetingRepresentation(vt, targetingValueEntryIdMapper, true)));
        }
        if (negativeValue != null) {
            List<V> versionedTargetings = valueModelProperty.get(negativeValue);
            checkState(versionedTargetings != null, "values of versionedTargeting must not be null");
            result.addAll(mapList(versionedTargetings,
                    vt -> toVersionedTargetingRepresentation(vt, targetingValueEntryIdMapper, false)));
        }
        return result;
    }

    public <T extends AdGroupAdditionalTargeting, V extends VersionedTargeting> void setVersionedAdditionalTargeting(
            Supplier<T> targetingConstructor, Supplier<V> valueConstructor,
            ModelProperty<T, List<V>> valueModelProperty,
            Map<String, Long> targetingValueEntryIdMapper,
            @Nullable List<VersionedTargetingRepresentation> versionedTargetingRepresentations) {
        if (versionedTargetingRepresentations == null) {
            return;
        }
        List<V> positiveValues = filterAndMapList(versionedTargetingRepresentations, t -> t.getPositiveValue() != null,
                t -> toVersionedTargetingValue(t, valueConstructor, targetingValueEntryIdMapper, true)
        );
        List<V> negativeValues = filterAndMapList(versionedTargetingRepresentations, t -> t.getNegativeValue() != null,
                t -> toVersionedTargetingValue(t, valueConstructor, targetingValueEntryIdMapper, false)
        );
        if (!positiveValues.isEmpty()) {
            targetingStore.addAdditionalTargetingWithValue(
                    targetingConstructor, valueModelProperty, true, positiveValues,
                    toVersionedTargetingOriginalValue(versionedTargetingRepresentations, true));
        }
        if (!negativeValues.isEmpty()) {
            targetingStore.addAdditionalTargetingWithValue(
                    targetingConstructor, valueModelProperty, false, negativeValues,
                    toVersionedTargetingOriginalValue(versionedTargetingRepresentations, false));
        }
    }

    private static <V extends VersionedTargeting> V toVersionedTargetingValue(
            VersionedTargetingRepresentation targetingRepresentation,
            Supplier<V> targetingValueConstructor,
            Map<String, Long> targetingValueEntryIdMapper,
            boolean isPositive) {
        String value = isPositive
                ? targetingRepresentation.getPositiveValue()
                : targetingRepresentation.getNegativeValue();

        //noinspection unchecked
        return (V) targetingValueConstructor.get()
                .withMaxVersion(addNumberDecimalPartIfNeed(targetingRepresentation.getMaxVersion()))
                .withMinVersion(addNumberDecimalPartIfNeed(targetingRepresentation.getMinVersion()))
                // приводим значение в нижний регистр, чтобы можно было получить entryId без учета регистра
                .withTargetingValueEntryId(targetingValueEntryIdMapper.get(value.toLowerCase()));
    }

    /**
     * Конвертируем в объект, который можно отдать фронту
     */
    private static List<VersionedTargetingInfo> toVersionedTargetingOriginalValue(
            List<VersionedTargetingRepresentation> versionedTargetingRepresentations, boolean isTargeting) {
        Function<VersionedTargetingRepresentation, String> valueGetter =
                representation -> isTargeting ? representation.getPositiveValue() : representation.getNegativeValue();

        return filterAndMapList(versionedTargetingRepresentations,
                representation -> valueGetter.apply(representation) != null,
                representation -> {
                    String value = valueGetter.apply(representation);
                    return new VersionedTargetingInfo()
                            .withMaxVersion(addNumberDecimalPartIfNeed(representation.getMaxVersion()))
                            .withMinVersion(addNumberDecimalPartIfNeed(representation.getMinVersion()))
                            .withValue(Objects.requireNonNull(value));
                });
    }

    /**
     * Добавляет к строке ".0", если в ней хранится целое положительное число
     * Нужно, чтобы привести значение версии к формату, который ожидает бэк
     * На гриде это делает фронт)
     */
    @Nullable
    static String addNumberDecimalPartIfNeed(@Nullable String value) {
        if (value == null) {
            return null;
        }

        if (NumberUtils.isDigits(value)) {
            return value + ".0";
        }

        return value;
    }

    private static <V extends VersionedTargeting> VersionedTargetingRepresentation toVersionedTargetingRepresentation(
            V versionedTargeting,
            Map<Long, String> targetingValueEntryIdMapper,
            boolean isTargeting) {
        checkState(targetingValueEntryIdMapper.containsKey(versionedTargeting.getTargetingValueEntryId()),
                "not found value for targetingValueEntryId = %s", versionedTargeting.getTargetingValueEntryId());
        var vtr = new VersionedTargetingRepresentation()
                .setMinVersion(versionedTargeting.getMinVersion())
                .setMaxVersion(versionedTargeting.getMaxVersion());
        var targetingValue = targetingValueEntryIdMapper.get(versionedTargeting.getTargetingValueEntryId());
        if (isTargeting) {
            return vtr.setPositiveValue(targetingValue);
        }
        return vtr.setNegativeValue(targetingValue);
    }
}
