package ru.yandex.direct.excel.processing.model.internalad;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.crypta.AudienceType;
import ru.yandex.direct.core.entity.retargeting.Constants;
import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.excel.processing.exception.ExcelValidationException;
import ru.yandex.direct.excel.processing.service.internalad.CryptaSegmentDictionariesService;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.retargeting.model.Goal.METRIKA_AUDIENCE_LOWER_BOUND;
import static ru.yandex.direct.excel.processing.validation.defects.ExcelDefectIds.CRYPTA_GOAL_NOT_FOUND;
import static ru.yandex.direct.excel.processing.validation.defects.ExcelDefectIds.MORE_THAN_ONE_GOAL_TYPE_IN_ONE_RULE;
import static ru.yandex.direct.excel.processing.validation.defects.ExcelDefectIds.SOCIAL_DEMO_GOAL_NOT_FOUND;
import static ru.yandex.direct.excel.processing.validation.defects.ExcelDefectIds.UNSUPPORTED_AUDIENCE_RULE_TYPE;
import static ru.yandex.direct.excel.processing.validation.defects.ExcelDefectIds.UNSUPPORTED_CRYPTA_RULE_TYPE;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class RetargetingConditionRepresentation {

    private static final Long SOCIAL_DEMO_GENDER_PARENT_ID = AudienceType.GENDER.getTypedValue();
    private static final Long SOCIAL_DEMO_AGE_PARENT_ID = AudienceType.AGE.getTypedValue();
    private static final Long SOCIAL_DEMO_INCOME_PARENT_ID = AudienceType.INCOME.getTypedValue();
    private static final String RETARGETING_CONDITION_NAME = "Условие ретаргетинга";
    private static final Set<GoalType> CRYPTA_GOAL_TYPES = Set.of(GoalType.FAMILY, GoalType.INTERESTS,
            GoalType.BEHAVIORS, GoalType.INTERNAL);

    private final List<Rule> rules;
    private final CryptaSegmentDictionariesService cryptaSegmentDictionariesService;

    public RetargetingConditionRepresentation(CryptaSegmentDictionariesService cryptaSegmentDictionariesService) {
        this(new ArrayList<>(), cryptaSegmentDictionariesService);
    }

    public RetargetingConditionRepresentation(List<Rule> retargetingConditionRules,
                                              CryptaSegmentDictionariesService cryptaSegmentDictionariesService) {
        this.cryptaSegmentDictionariesService = cryptaSegmentDictionariesService;
        this.rules = retargetingConditionRules;
    }

    @Nullable
    public RetargetingCondition getRetargetingCondition() {
        if (rules.size() == 0) {
            return null;
        }

        return (RetargetingCondition) new RetargetingCondition()
                .withName(RETARGETING_CONDITION_NAME)
                .withDeleted(false)
                .withType(ConditionType.interests)
                .withRules(rules);
    }

    public List<Rule> getGoalContext() {
        return filterList(rules, rule -> isGoalType(rule, GoalType.GOAL));
    }

    public void setGoalContext(@Nullable List<Rule> goalContextRules) {
        if (goalContextRules != null) {
            rules.addAll(goalContextRules);
        }
    }

    public List<Long> getAudience() {
        return getAudienceBase(RuleType.OR);
    }

    public List<Long> getAudienceNot() {
        return getAudienceBase(RuleType.NOT);
    }

    private List<Long> getAudienceBase(RuleType ruleType) {
        List<Rule> unsupportedAudienceRules = filterList(rules,
                rule -> isGoalType(rule, GoalType.AUDIENCE) && rule.getType() == RuleType.ALL);
        if (!unsupportedAudienceRules.isEmpty()) {
            throw ExcelValidationException.create(UNSUPPORTED_AUDIENCE_RULE_TYPE);
        }
        List<Rule> audienceRules = filterList(rules,
                rule -> isGoalType(rule, GoalType.AUDIENCE) && rule.getType() == ruleType);
        if (audienceRules.size() <= 1) {
            return mapList(rulesToGoalIds(audienceRules), audienceId -> {
                checkState(audienceId >= METRIKA_AUDIENCE_LOWER_BOUND, "Unexpected audienceId: " + audienceId
                        + ", audienceId must be greater than: " + METRIKA_AUDIENCE_LOWER_BOUND);
                return audienceId - METRIKA_AUDIENCE_LOWER_BOUND;
            });
        } else {
            throw ExcelValidationException.create(UNSUPPORTED_AUDIENCE_RULE_TYPE);
        }
    }

    public void setAudience(List<Long> goals) {
        if (goals.size() != 0) {
            setAudienceBase(goals, RuleType.OR);
        }
    }

    public void setAudienceNot(List<Long> goals) {
        if (goals.size() != 0) {
            setAudienceBase(goals, RuleType.NOT);
        }
    }

    private void setAudienceBase(List<Long> goalIds, RuleType ruleType) {
        List<Goal> goals = StreamEx.of(goalIds)
                .map(goal -> goal < METRIKA_AUDIENCE_LOWER_BOUND ? goal + METRIKA_AUDIENCE_LOWER_BOUND : goal)
                .map(goal -> (Goal) new Goal().withTime(Constants.AUDIENCE_TIME_VALUE).withId(goal))
                .toList();
        rules.addAll(List.of(new Rule()
                .withType(ruleType)
                .withGoals(goals)));
    }

    public List<CryptaSegment> getCrypta() {
        return getCryptaBase(RuleType.OR);
    }

    public List<CryptaSegment> getCryptaNot() {
        return getCryptaBase(RuleType.NOT);
    }

    private List<CryptaSegment> getCryptaBase(RuleType ruleType) {
        List<Rule> unsupportedCryptaRules = filterList(rules,
                rule -> isCryptaGoalType(rule) && rule.getType() == RuleType.ALL);
        if (!unsupportedCryptaRules.isEmpty()) {
            throw ExcelValidationException.create(UNSUPPORTED_CRYPTA_RULE_TYPE);
        }

        List<Rule> cryptaRules = filterList(rules, rule -> isCryptaGoalType(rule) && rule.getType() == ruleType);
        if (cryptaRules.size() <= 1) {
            List<Long> goalIds = rulesToGoalIds(cryptaRules);
            return mapList(goalIds, id -> {
                Goal value = cryptaSegmentDictionariesService.getCryptaByGoalId(id);
                //noinspection ConstantConditions
                return CryptaSegment.create(value.getKeyword(), value.getKeywordValue());
            });
        } else {
            throw ExcelValidationException.create(UNSUPPORTED_CRYPTA_RULE_TYPE);
        }
    }

    public void setCrypta(List<CryptaSegment> crypta) {
        if (crypta.size() != 0) {
            setCryptaBase(crypta, RuleType.OR);
        }
    }

    public void setCryptaNot(List<CryptaSegment> cryptaNot) {
        if (cryptaNot.size() != 0) {
            setCryptaBase(cryptaNot, RuleType.NOT);
        }
    }

    private void setCryptaBase(List<CryptaSegment> cryptaSegments, RuleType ruleType) {
        List<Goal> goals = mapList(cryptaSegments, c -> {
            Goal value = cryptaSegmentDictionariesService.getGoalIdByKeywordAndType(c.getKeywordId(), c.getSegmentId());
            if (value == null) {
                throw ExcelValidationException.create(CRYPTA_GOAL_NOT_FOUND, cryptaSegments);
            }
            return (Goal) new Goal().withTime(0).withId(value.getId());
        });

        rules.add(new Rule()
                .withType(ruleType)
                .withGoals(goals));
    }

    public List<String> getSocialDemoGender() {
        return getSocialDemoValues(SOCIAL_DEMO_GENDER_PARENT_ID);
    }

    public List<String> getSocialDemoAge() {
        return getSocialDemoValues(SOCIAL_DEMO_AGE_PARENT_ID);
    }

    public List<String> getSocialDemoIncome() {
        return getSocialDemoValues(SOCIAL_DEMO_INCOME_PARENT_ID);
    }

    private List<String> getSocialDemoValues(Long parentId) {
        List<Rule> socialDemoRules = filterList(rules, rule -> isGoalType(rule, GoalType.SOCIAL_DEMO));

        if (socialDemoRules.size() == 0) {
            return emptyList();
        } else {
            List<Long> goalIdsByParentId = cryptaSegmentDictionariesService.getCryptaGoalIdsByParentId(parentId);
            List<Rule> rulesWithCorrectGoal = filterList(socialDemoRules, rule -> isInList(rule, goalIdsByParentId));

            //noinspection ConstantConditions
            return StreamEx.of(rulesWithCorrectGoal)
                    .map(Rule::getGoals)
                    .flatMap(List::stream)
                    .map(GoalBase::getId)
                    .distinct()
                    .map(cryptaSegmentDictionariesService::getCryptaByGoalId)
                    .map(GoalBase::getName)
                    .toList();
        }
    }

    public void setSocialDemoGender(List<String> values) {
        setSocialDemoValues(values, SOCIAL_DEMO_GENDER_PARENT_ID);
    }

    public void setSocialDemoAge(List<String> values) {
        setSocialDemoValues(values, SOCIAL_DEMO_AGE_PARENT_ID);
    }

    public void setSocialDemoIncome(List<String> values) {
        setSocialDemoValues(values, SOCIAL_DEMO_INCOME_PARENT_ID);
    }

    private void setSocialDemoValues(List<String> values, Long parentId) {
        if (values.isEmpty()) {
            return;
        }

        List<Goal> goals = mapList(values, value -> {
            Goal goal = cryptaSegmentDictionariesService.getGoalIdByParentIdAndName(parentId, value);
            if (goal == null) {
                throw ExcelValidationException.create(SOCIAL_DEMO_GOAL_NOT_FOUND, values);
            }

            return (Goal) new Goal()
                    .withTime(0)
                    .withId(goal.getId());
        });

        rules.add(new Rule()
                .withType(RuleType.OR)
                .withGoals(goals));
    }

    private static boolean isGoalType(Rule rule, GoalType goalType) {
        Set<GoalType> goalTypes = listToSet(rule.getGoals(), GoalBase::getType);
        if (goalTypes.contains(goalType)) {
            // для крипты могут быть разные типы
            if (goalTypes.size() != 1 && !isAllCryptaGoalType(goalTypes)) {
                throw ExcelValidationException.create(MORE_THAN_ONE_GOAL_TYPE_IN_ONE_RULE);
            }
            return true;
        }
        return false;
    }

    private static boolean isCryptaGoalType(Rule rule) {
        return CRYPTA_GOAL_TYPES.stream()
                .anyMatch(type -> isGoalType(rule, type));
    }

    /**
     * Являются ли все переданные типы типами Крипты
     */
    private static boolean isAllCryptaGoalType(Set<GoalType> goalTypes) {
        return CRYPTA_GOAL_TYPES.containsAll(goalTypes);
    }

    private static boolean isInList(Rule rule, List<Long> goalIds) {
        for (Goal goal : rule.getGoals()) {
            if (!goalIds.contains(goal.getId())) {
                return false;
            }
        }
        return true;
    }

    private static List<Long> rulesToGoalIds(List<Rule> rules) {
        return StreamEx.of(rules)
                .flatCollection(Rule::getGoals)
                .map(GoalBase::getId)
                .toList();
    }
}
