package ru.yandex.direct.excel.processing.model.internalad;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;


@ParametersAreNonnullByDefault
public class RulesConverter {

    private RulesConverter() {
    }

    /**
     * Конвертируем строку вида (100500:2)&(~20141023:21)&(20141020:1|20141025:2)
     * из String в List<Rule>, где выражение в скобках является правилом типа ALL, NOT или OR
     */
    public static List<Rule> convertStringToRules(String goalContextRules) {
        String[] stringOfRules = goalContextRules
                .replace("(", "")
                .replace(")", "").split("&");
        return StreamEx.of(stringOfRules)
                .map(RulesConverter::convertStringToRule)
                .toList();
    }

    private static Rule convertStringToRule(String stringRule) {
        if (stringRule.contains("~")) {
            return parseNot(stringRule);
        } else {
            return parseOr(stringRule);
        }
    }

    private static Rule parseNot(String stringRule) {
        stringRule = stringRule.replace("~", "");
        return new Rule()
                .withType(RuleType.NOT)
                .withGoals(List.of(createGoal(stringRule)));
    }

    private static Rule parseOr(String stringRule) {
        String[] subrule = stringRule.split("\\|");
        return new Rule()
                .withType(RuleType.OR)
                .withGoals(StreamEx.of(subrule)
                        .map(RulesConverter::createGoal)
                        .toList());
    }

    private static Goal createGoal(String stringGoal) {
        String[] goal = stringGoal.split(":");
        return (Goal) new Goal()
                .withId(Long.valueOf(goal[0]))
                .withTime(Integer.valueOf(goal[1]));
    }

    /**
     * Конвертируем набор правил в строку вида (100500:2)&(~20141023:21)&(20141020:1|20141025:2)
     * из List<Rule> в String, где выражение в скобках является правилом типа ALL, NOT или OR
     */
    public static String convertRulesToString(List<Rule> rules) {
        return StreamEx.of(rules)
                .map(RulesConverter::convertRuleToString)
                .joining("&");
    }

    private static String convertRuleToString(Rule rule) {
        StringBuilder stringRule = null;
        if (rule.getType() == RuleType.OR || rule.getType() == RuleType.ALL) {
            stringRule = new StringBuilder("(");
        } else if (rule.getType() == RuleType.NOT) {
            stringRule = new StringBuilder("(~");
        }
        for (int i = 0; i < rule.getGoals().size(); i++) {
            stringRule.append(createStringGoal(rule.getGoals().get(i)));
            if (i != rule.getGoals().size() - 1) {
                if (rule.getType() == RuleType.OR) {
                    stringRule.append("|");
                } else if (rule.getType() == RuleType.ALL) {
                    stringRule.append(")&(");
                } else if (rule.getType() == RuleType.NOT) {
                    stringRule.append(")&(~");
                }
            } else {
                stringRule.append(")");
            }
        }
        return stringRule.toString();
    }

    private static String createStringGoal(Goal goal) {
        return goal.getId() + ":" + goal.getTime();
    }
}
