package ru.yandex.direct.excel.processing.model.internalad;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingJoinType;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingMode;
import ru.yandex.direct.excelmapper.mappers.YesNoBooleanExcelMapper;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingJoinType.ALL;
import static ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingJoinType.ANY;
import static ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingMode.FILTERING;
import static ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingMode.TARGETING;

@ParametersAreNonnullByDefault
public class TargetingStore {
    private final Map<TargetingKey, AdGroupAdditionalTargeting> targetingMap;

    /**
     * Хранит значения таргетингов прочитанных из excel файла без конвертации во внутренние модели
     * Нужно, чтобы можно было легко вернуть фронту, то что прочитали из файла
     */
    private final Map<TargetingKey, Object> targetingOriginalValueFromExcelMap;

    public TargetingStore() {
        // хотим сохранить порядок добавления таргетингов для показа на фронте
        this.targetingMap = new LinkedHashMap<>();
        this.targetingOriginalValueFromExcelMap = new HashMap<>();
    }

    TargetingStore(Collection<? extends AdGroupAdditionalTargeting> additionalTargetings) {
        this.targetingMap = StreamEx.of(additionalTargetings)
                .mapToEntry(TargetingKey::new, identity())
                .mapValues(AdGroupAdditionalTargeting.class::cast)
                .toImmutableMap();
        this.targetingOriginalValueFromExcelMap = null;
    }

    @Nullable
    public <T extends AdGroupAdditionalTargeting> T getAdGroupAdditionalTargeting(Class<T> targetingClass,
                                                                                  boolean isTargeting) {
        AdGroupAdditionalTargetingJoinType joinType = getDefaultJoinTypeByTargetingMode(isTargeting);
        return getAdGroupAdditionalTargeting(targetingClass, isTargeting, joinType);
    }

    @Nullable
    public <T extends AdGroupAdditionalTargeting> T getAdGroupAdditionalTargeting(
            Class<T> targetingClass,
            boolean isTargeting,
            AdGroupAdditionalTargetingJoinType joinType) {
        return getAdGroupAdditionalTargeting(targetingClass, isTargeting ? TARGETING : FILTERING, joinType);
    }

    @Nullable
    public Boolean getBooleanAdditionalTargeting(Class<? extends AdGroupAdditionalTargeting> targetingClass) {
        if (getAdGroupAdditionalTargeting(targetingClass, true) != null) {
            return Boolean.TRUE;
        } else if (getAdGroupAdditionalTargeting(targetingClass, false) != null) {
            return Boolean.FALSE;
        }
        return null;
    }

    @Nullable
    @SuppressWarnings("unchecked")
    private <T extends AdGroupAdditionalTargeting> T getAdGroupAdditionalTargeting(
            Class<T> targetingClass,
            AdGroupAdditionalTargetingMode mode,
            AdGroupAdditionalTargetingJoinType joinType) {
        return (T) this.targetingMap.get(new TargetingKey(targetingClass, mode, joinType));
    }

    public <T extends AdGroupAdditionalTargeting> void addBooleanAdditionalTargeting(@Nullable Boolean targeting,
                                                                                     Supplier<T> constructor) {
        if (targeting == null) {
            return;
        }

        T additionalTargeting = createTargetingAndFillCommonFields(constructor, targeting);
        addAdGroupAdditionalTargetingInStore(additionalTargeting, YesNoBooleanExcelMapper.booleanToString(targeting));
    }

    public <T extends AdGroupAdditionalTargeting, V> void addAdditionalTargetingWithValue(
            Supplier<T> constructor, ModelProperty<T, V> valueModelProperty, boolean isTargeting,
            @Nullable V value, Object originalValueFromExcel) {
        AdGroupAdditionalTargetingJoinType joinType = getDefaultJoinTypeByTargetingMode(isTargeting);
        addAdditionalTargetingWithValue(constructor, valueModelProperty, isTargeting, joinType, value,
                originalValueFromExcel);
    }

    public <T extends AdGroupAdditionalTargeting, V> void addAdditionalTargetingWithValue(
            Supplier<T> constructor, ModelProperty<T, V> valueModelProperty,
            boolean isTargeting, AdGroupAdditionalTargetingJoinType joinType,
            @Nullable V value, Object originalValueFromExcel) {
        if (value == null) {
            return;
        }

        T additionalTargeting = createTargetingAndFillCommonFields(constructor, isTargeting, joinType);
        valueModelProperty.set(additionalTargeting, value);
        addAdGroupAdditionalTargetingInStore(additionalTargeting, originalValueFromExcel);
    }

    public Collection<AdGroupAdditionalTargeting> getTargetings() {
        return targetingMap.values();
    }

    @Nullable
    public Object getTargetingOriginalValueFromExcel(AdGroupAdditionalTargeting additionalTargeting) {
        checkState(targetingOriginalValueFromExcelMap != null, "targetingWithOriginalValueMap not init");
        return targetingOriginalValueFromExcelMap.get(new TargetingKey(additionalTargeting));
    }

    /**
     * Сохраняет в хранилище полученный объект таргетинга и значение этого таргетинга прочитанные как есть из excel
     */
    private void addAdGroupAdditionalTargetingInStore(AdGroupAdditionalTargeting additionalTargeting,
                                                      Object originalValueFromExcel) {
        var targetingKey = new TargetingKey(additionalTargeting);
        targetingMap.put(targetingKey, additionalTargeting);
        targetingOriginalValueFromExcelMap.put(targetingKey, originalValueFromExcel);
    }

    private static <T extends AdGroupAdditionalTargeting> T createTargetingAndFillCommonFields(Supplier<T> constructor,
                                                                                               boolean isTargeting) {
        AdGroupAdditionalTargetingJoinType joinType = getDefaultJoinTypeByTargetingMode(isTargeting);
        return createTargetingAndFillCommonFields(constructor, isTargeting, joinType);
    }

    private static <T extends AdGroupAdditionalTargeting> T createTargetingAndFillCommonFields(
            Supplier<T> constructor,
            boolean isTargeting,
            AdGroupAdditionalTargetingJoinType joinType) {
        AdGroupAdditionalTargetingMode targetingMode = isTargeting ? TARGETING : FILTERING;
        T adGroupAdditionalTargeting = constructor.get();
        adGroupAdditionalTargeting
                .withJoinType(joinType)
                .withTargetingMode(targetingMode);
        return adGroupAdditionalTargeting;
    }

    public static AdGroupAdditionalTargetingJoinType getDefaultJoinTypeByTargetingMode(boolean isTargeting) {
        // По умолчанию значению TARGETING соответствует ANY (Or в интерфейсе), FILTERING - ALL (Not в интерфейсе).
        // Для таргетингов DesktopInstalledApps и MobileInstalledApps для значения TARGETING возможен еще ALL
        // (And в интерфейсе).
        return isTargeting ? ANY : ALL;
    }

    private static class TargetingKey {
        private final Class<? extends AdGroupAdditionalTargeting> clazz;
        private final AdGroupAdditionalTargetingMode targetingMode;
        private final AdGroupAdditionalTargetingJoinType joinType;

        private <T extends AdGroupAdditionalTargeting> TargetingKey(T targeting) {
            this(targeting.getClass(), targeting.getTargetingMode(), targeting.getJoinType());
        }

        private TargetingKey(Class<? extends AdGroupAdditionalTargeting> clazz,
                             AdGroupAdditionalTargetingMode targetingMode,
                             AdGroupAdditionalTargetingJoinType joinType) {
            this.clazz = clazz;
            this.targetingMode = targetingMode;
            this.joinType = joinType;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            TargetingKey key = (TargetingKey) o;
            return clazz.equals(key.clazz) &&
                    targetingMode == key.targetingMode &&
                    joinType == key.joinType;
        }

        @Override
        public int hashCode() {
            return Objects.hash(clazz, targetingMode, joinType);
        }
    }
}
