package ru.yandex.direct.excel.processing.model.internalad.mappers;

import java.util.Comparator;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.model.InternalModerationInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceChoice;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.excel.processing.model.internalad.InternalBannerRepresentation;
import ru.yandex.direct.excelmapper.ExcelMapper;
import ru.yandex.direct.excelmapper.mappers.HyperlinkExcelMapper;
import ru.yandex.direct.excelmapper.mappers.ObjectExcelMapper;
import ru.yandex.direct.utils.CollectionUtils;

import static java.util.Collections.emptyMap;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.excelmapper.ExcelMappers.maybeHyperlinkOrStringMapper;
import static ru.yandex.direct.excelmapper.ExcelMappers.maybeLongMapper;
import static ru.yandex.direct.excelmapper.ExcelMappers.maybeStringMapper;
import static ru.yandex.direct.excelmapper.ExcelMappers.maybeYesNoBooleanMapper;
import static ru.yandex.direct.excelmapper.ExcelMappers.modelMapper;
import static ru.yandex.direct.excelmapper.ExcelMappers.objectMapper;
import static ru.yandex.direct.excelmapper.ExcelMappers.stringMapper;
import static ru.yandex.direct.excelmapper.ExcelMappers.yesNoBooleanMapper;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class BannerMappers {

    public static final String AD_GROUP_NAME_TITLE = "Название группы";
    public static final String AD_GROUP_ID_TITLE = "ID группы";
    public static final String BANNER_NAME_TITLE = "Название объявления";
    public static final String BANNER_ID_TITLE = "ID объявления";
    public static final String STATUS_SHOW_TITLE = "Включен ли";

    public static final String STATUS_SEND_TO_MODERATION_TITLE = "Отправить на модерацию";
    public static final String STATUS_SHOW_AFTER_MODERATION_TITLE = "Включен ли после модерации";
    public static final String IS_SECRET_AD_TITLE = "NDA запуск";
    public static final String TICKET_URL_TITLE = "Согласование (тикет)";
    public static final String CUSTOM_COMMENT_TITLE = "Комментарий для модератора";

    private static final ExcelMapper<InternalBanner> INTERNAL_BANNER_MAPPER = modelMapper(InternalBanner::new)
            .field(InternalBanner.ID, maybeLongMapper(BANNER_ID_TITLE))
            .field(InternalBanner.DESCRIPTION, maybeStringMapper(BANNER_NAME_TITLE))
            .field(InternalBanner.STATUS_SHOW, yesNoBooleanMapper(STATUS_SHOW_TITLE))
            .build();

    private static final ExcelMapper<InternalModerationInfo> MODERATION_INFORMATION_MAPPER =
            modelMapper(InternalModerationInfo::new)
                    .fieldWithDefaultValueForGetter(InternalModerationInfo.SEND_TO_MODERATION,
                            yesNoBooleanMapper(STATUS_SEND_TO_MODERATION_TITLE), false)
                    .field(InternalModerationInfo.STATUS_SHOW_AFTER_MODERATION,
                            yesNoBooleanMapper(STATUS_SHOW_AFTER_MODERATION_TITLE))
                    .fieldWithDefaultValueForSetter(InternalModerationInfo.IS_SECRET_AD,
                            maybeYesNoBooleanMapper(IS_SECRET_AD_TITLE), false)
                    .field(InternalModerationInfo.TICKET_URL, stringMapper(TICKET_URL_TITLE))
                    .field(InternalModerationInfo.CUSTOM_COMMENT, maybeStringMapper(CUSTOM_COMMENT_TITLE))
                    .build();

    private BannerMappers() {
    }

    public static ExcelMapper<InternalBannerRepresentation> getInternalBannersMapperOnlyForRead(
            List<ResourceInfo> resources, boolean isModerated) {
        // передаем пустую мапу, т.к. она нужна только для записи в эксель, а тут мы хотим только читать из экселя
        return getInternalBannersMapper(resources, emptyMap(), isModerated);
    }

    /**
     * Возвращает маппер для объявлений внутренней рекламы
     * Для картиночных переменных маппер добавдяет имя файла к хэшу изображения
     * Если у поля есть ограничивающие значения, то нужно добавить в {@link BannerMapperSettings#getColumnsWithChoices}
     */
    public static ExcelMapper<InternalBannerRepresentation> getInternalBannersMapper(
            List<ResourceInfo> resources,
            Map<String, String> imageUrlByHashes,
            boolean isModerated) {
        ObjectExcelMapper.ObjectExcelMapperBuilder<InternalBannerRepresentation> mapperBuilder =
                objectMapper(InternalBannerRepresentation::new)
                        .field(InternalBannerRepresentation::getAdGroupId, InternalBannerRepresentation::setAdGroupId,
                                maybeLongMapper(AD_GROUP_ID_TITLE))
                        .field(InternalBannerRepresentation::getAdGroupName,
                                InternalBannerRepresentation::setAdGroupName,
                                stringMapper(AD_GROUP_NAME_TITLE))
                        .field(InternalBannerRepresentation::getBanner, InternalBannerRepresentation::setBanner,
                                INTERNAL_BANNER_MAPPER);

        // добавляем информацию о модерации
        if (isModerated) {
            mapperBuilder.field(InternalBannerRepresentation::getModerationInfo,
                    InternalBannerRepresentation::setModerationInfo,
                    MODERATION_INFORMATION_MAPPER);
        }

        // добавляем маперы для переменных
        if (isNotEmpty(resources)) {
            resources.stream()
                    .sorted(Comparator.comparing(ResourceInfo::getPosition))
                    .forEach(resource ->
                            mapperBuilder.field(
                                    obj -> variableValueGetter(obj, resource, imageUrlByHashes),
                                    (obj, value) -> variableValueSetter(obj, value, resource),
                                    maybeHyperlinkOrStringMapper(getResourceDescription(resource))
                            )
                    );
        }

        return mapperBuilder.build();
    }

    private static String getResourceDescription(ResourceInfo resource) {
        return nvl(resource.getLabel(), resource.getId().toString());
    }

    @Nullable
    static String variableValueGetter(InternalBannerRepresentation representation,
                                      ResourceInfo resource,
                                      Map<String, String> imageUrlByHashes) {
        String value = representation.getTemplateVariableValue(resource.getId());
        // Для картинок добавляем имя файла если есть
        if (isExistingImage(representation, resource, imageUrlByHashes)) {
            return HyperlinkExcelMapper.makeHyperlink(imageUrlByHashes.get(value), value);
        }

        List<ResourceChoice> choices = resource.getChoices();
        if (CollectionUtils.isEmpty(choices)) {
            return value;
        }

        // Для ресурсов с выбором заменяем на отображаемое название варианта
        return choices.stream()
                .filter(c -> c.getValue().equals(value))
                .findAny()
                .map(ResourceChoice::getDisplay)
                .orElse(value);
    }

    static void variableValueSetter(InternalBannerRepresentation representation,
                                    @Nullable String value,
                                    ResourceInfo resource) {
        List<ResourceChoice> choices = resource.getChoices();

        if (resource.getType() == ResourceType.IMAGE && StringUtils.isNotEmpty(value)) {
            value = getValueForImage(value);
        } else if (!CollectionUtils.isEmpty(choices)) {
            // Для ресурсов с выбором заменяем отображаемое название на значение, которое нужно сохранить в базу
            String oldValue = value;
            value = choices.stream()
                    .filter(c -> c.getDisplay().equals(oldValue))
                    .findAny()
                    .map(ResourceChoice::getValue)
                    .orElse(oldValue);
        }

        representation.setTemplateVariableValue(resource.getId(), value);
    }

    /**
     * Для картинок в экселе ожидаем следующие значения:
     * 1. Имя файла. Потом на втором шаге импорта можно выбрать и залить этот файл
     * 2. Гиперссылка с хешом и ссылкой на картинку в аватарнице.
     * 3. Урл на картинку. На втором шаге импорта фронт загрузить картинку по урлу
     */
    private static String getValueForImage(String value) {
        value = value.strip();

        Pair<String, String> maybeHyperlink = HyperlinkExcelMapper.getLinkAndTitle(value);
        if (maybeHyperlink != null) {
            return maybeHyperlink.getRight();
        }

        return value;
    }

    private static boolean isExistingImage(InternalBannerRepresentation representation,
                                           ResourceInfo resource,
                                           Map<String, String> imageUrlByHashes) {
        String value = representation.getTemplateVariableValue(resource.getId());
        return resource.getType() == ResourceType.IMAGE
                && StringUtils.isNotEmpty(value) && imageUrlByHashes.containsKey(value);
    }
}
