package ru.yandex.direct.excel.processing.model.internalad.mappers;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.excel.processing.model.internalad.SheetDescriptor;
import ru.yandex.direct.excelmapper.ExcelMapper;
import ru.yandex.direct.excelmapper.Height;
import ru.yandex.direct.excelmapper.MapperMeta;
import ru.yandex.direct.excelmapper.MapperUtils;
import ru.yandex.direct.excelmapper.ReadResult;
import ru.yandex.direct.excelmapper.SheetRange;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

@ParametersAreNonnullByDefault
public class SheetDescriptorMapper implements ExcelMapper<SheetDescriptor> {

    private static final String CAMPAIGN_ID_TITLE = "Кампания";
    private static final String PLACE_ID_TITLE = "Место показа";
    private static final String TEMPLATE_ID_TITLE = "Шаблон объявления";

    private static final int CAMPAIGN_ID_ROW = 0;
    private static final int PLACE_ID_ROW = 1;
    private static final int TEMPLATE_ID_ROW = 2;

    private final MapperMeta meta;

    public enum SheetType {AD_GROUP_LIST_SHEET, BANNER_SHEET}

    private final SheetType sheetType;

    public SheetDescriptorMapper(SheetType sheetType) {
        List<String> titles = List.of("Наименование параметра", "Значение");
        this.meta = new MapperMeta(titles, Height.dynamic(), Set.copyOf(titles));
        this.sheetType = sheetType;
    }

    @Override
    public MapperMeta getMeta() {
        return meta;
    }

    @Override
    public int write(SheetRange sheetRange, @Nullable SheetDescriptor value) {
        checkArgument(value != null);
        sheetRange.getCellForWrite(0, 0).setCellValue(CAMPAIGN_ID_TITLE);
        sheetRange.getCellForWrite(0, 1).setCellValue(value.getCampaignId().toString());
        sheetRange.getCellForWrite(1, 0).setCellValue(PLACE_ID_TITLE);
        sheetRange.getCellForWrite(1, 1).setCellValue(value.getPlaceId().toString());
        if (value.getTemplateId() != null) {
            sheetRange.getCellForWrite(2, 0).setCellValue(TEMPLATE_ID_TITLE);
            sheetRange.getCellForWrite(2, 1).setCellValue(value.getTemplateId().toString());
            return 3;
        }

        return 2;
    }

    @Override
    public ReadResult<SheetDescriptor> read(SheetRange sheetRange) {
        var sheetDescriptor = new SheetDescriptor();
        sheetDescriptor.setCampaignId(tryReadLongValue(sheetRange, CAMPAIGN_ID_ROW, CAMPAIGN_ID_TITLE));
        sheetDescriptor.setPlaceId(tryReadLongValue(sheetRange, PLACE_ID_ROW, PLACE_ID_TITLE));

        if (sheetType == SheetType.BANNER_SHEET) {
            sheetDescriptor.setTemplateId(tryReadLongValue(sheetRange, TEMPLATE_ID_ROW, TEMPLATE_ID_TITLE));
            return new ReadResult<>(sheetDescriptor, 3);
        }
        return new ReadResult<>(sheetDescriptor, 2);
    }

    private Long tryReadLongValue(SheetRange sheetRange, int row, String columnName) {
        try {
            String stringCellValue = tryReadStringCellValue(sheetRange, row);
            return Long.valueOf(stringCellValue);
        } catch (NumberFormatException e) {
            sheetRange.reportCantReadFormat(List.of(columnName), row, 1);
        }
        // недостижимое место
        throw new IllegalStateException();
    }

    @Override
    public boolean canStartReading(SheetRange sheetRange) {
        return isNotBlank(tryReadStringCellValue(sheetRange, 0))
                && isNotBlank(tryReadStringCellValue(sheetRange, 1))
                && isNotBlank(tryReadStringCellValue(sheetRange, 2));
    }

    private String tryReadStringCellValue(SheetRange sheetRange, int row) {
        return MapperUtils.tryReadStringCellValueOrReportCantRead(sheetRange, row, 1, meta.getColumns());
    }

}
