package ru.yandex.direct.excel.processing.service.actreport

import org.apache.poi.ss.usermodel.BorderStyle
import org.apache.poi.ss.usermodel.CellStyle
import org.apache.poi.ss.usermodel.CellType
import org.apache.poi.ss.usermodel.Font
import org.apache.poi.ss.usermodel.HorizontalAlignment
import org.apache.poi.ss.usermodel.Row
import org.apache.poi.ss.usermodel.Sheet
import org.apache.poi.ss.usermodel.Workbook
import org.apache.poi.ss.util.CellRangeAddress
import org.apache.poi.ss.util.RegionUtil.setBorderBottom
import org.apache.poi.ss.util.RegionUtil.setBorderLeft
import org.apache.poi.ss.util.RegionUtil.setBorderRight
import org.apache.poi.ss.util.RegionUtil.setBorderTop
import org.apache.poi.xssf.usermodel.XSSFWorkbook
import ru.yandex.direct.excel.processing.model.actreport.ActReportExcelExportData
import kotlin.reflect.KProperty1

private data class PropField<P>(
    val property: P,
    val name: String
)

class ActReportExcelExportService {
    companion object {
        private const val COST_TITLE = "Расходы"
    }

    fun createReport(actItems: List<ActReportExcelExportData>): XSSFWorkbook {
        val statFields = getStatFields()
        val costFields = getCostFields(actItems)
        return XSSFWorkbook().apply {
            createSheet("report").apply {
                writeHeader(1, statFields, costFields)
                writeValues(3, actItems, statFields + costFields)
                autosizeColumns(0, statFields.size + costFields.size - 1)
            }
        }
    }

    private fun Sheet.writeHeader(
        rowIndex: Int,
        statFields: List<PropField<out KProperty1<ActReportExcelExportData, Any>>>,
        costFields: List<PropField<out KProperty1<ActReportExcelExportData, Any>>>
    ) {
        val allFields = statFields + costFields

        val centerAlignedStyle = workbook.createStyle(alignment = HorizontalAlignment.CENTER)
        createRow(rowIndex)
            .writeCell(statFields.size, COST_TITLE, centerAlignedStyle)
        mergeColumns(rowIndex, statFields.size, allFields.size - 1)

        val leftAlignedStyle = workbook.createStyle(alignment = HorizontalAlignment.LEFT)
        createRow(rowIndex + 1).apply {
            allFields.forEachIndexed { i, field ->
                writeCell(i, field.name, leftAlignedStyle)
            }
        }
        // inner borders
        for (i in rowIndex..rowIndex + 1) {
            for (j in allFields.indices) {
                setBorderStyle(CellRangeAddress(i, i, j, j), BorderStyle.THIN)
            }
        }
        // outer borders
        setBorderStyle(CellRangeAddress(rowIndex, rowIndex + 1, 0, allFields.size - 1), BorderStyle.MEDIUM)
    }

    private fun Sheet.writeValues(
        rowIndex: Int, items: List<ActReportExcelExportData>,
        fields: List<PropField<out KProperty1<ActReportExcelExportData, Any>>>
    ) {
        val rightAlignedStyle = workbook.createStyle(alignment = HorizontalAlignment.RIGHT)
        items.forEachIndexed { i, item ->
            writeRow(rowIndex + i, item, fields, rightAlignedStyle)
        }
    }

    private fun Sheet.writeRow(
        rowIndex: Int, item: ActReportExcelExportData,
        fields: List<PropField<out KProperty1<ActReportExcelExportData, Any>>>,
        style: CellStyle
    ) {
        createRow(rowIndex).apply {
            fields.forEachIndexed { i, field ->
                writeCell(i, field.property.get(item), style)
            }
        }
    }

    private fun Row.writeCell(colIndex: Int, value: Any, style: CellStyle) {
        createCell(colIndex).apply {
            when (value) {
                is Long -> {
                    setCellType(CellType.NUMERIC)
                    setCellValue(value.toDouble())
                }
                is String -> {
                    setCellType(CellType.STRING)
                    setCellValue(value)
                }
            }
            cellStyle = style
        }
    }

    private fun Sheet.mergeColumns(rowIndex: Int, colIndexFrom: Int, colIndexTo: Int) {
        addMergedRegion(CellRangeAddress(rowIndex, rowIndex, colIndexFrom, colIndexTo))
    }

    private fun Sheet.autosizeColumns(colIndexFrom: Int, colIndexTo: Int) {
        for (i in colIndexFrom..colIndexTo) {
            autoSizeColumn(i)
        }
    }

    private fun getStatFields(): List<PropField<out KProperty1<ActReportExcelExportData, Any>>> {
        return listOf(
            PropField(ActReportExcelExportData::date, "Дата"),
            PropField(ActReportExcelExportData::cid, "Номер РК"),
            PropField(ActReportExcelExportData::shows, "Показы"),
            PropField(ActReportExcelExportData::clicks, "Клики"),
            PropField(ActReportExcelExportData::conversions, "Конверсии")
        )
    }

    private fun getCostFields(
        actItems: List<ActReportExcelExportData>
    ): List<PropField<KProperty1<ActReportExcelExportData, Long>>> {
        val fixFields = listOf(
            PropField(ActReportExcelExportData::total, "Всего"),
            PropField(ActReportExcelExportData::money, "Деньги"),
            PropField(ActReportExcelExportData::bonus, "Бонусы"),
            PropField(ActReportExcelExportData::backs, "Корректировки"),
        )
        val optFields = listOf(
            PropField(ActReportExcelExportData::promo, "Промокоды"),
            PropField(ActReportExcelExportData::sales, "Скидки"),
            PropField(ActReportExcelExportData::certs, "Сертификаты")
        )
        return fixFields + optFields.filter { field -> actItems.any { field.property.get(it) != 0L } }
    }

    private fun Sheet.setBorderStyle(region: CellRangeAddress, style: BorderStyle) {
        setBorderBottom(style, region, this)
        setBorderRight(style, region, this)
        setBorderLeft(style, region, this)
        setBorderTop(style, region, this)
    }

    private fun Workbook.createDefaultFont(): Font {
        return createFont().apply {
            fontHeightInPoints = 12
        }
    }

    private fun Workbook.createStyle(
        alignment: HorizontalAlignment,
        font: Font = createDefaultFont()
    ): CellStyle {
        return createCellStyle().apply {
            setAlignment(alignment)
            setFont(font)
        }
    }
}
