package ru.yandex.direct.excel.processing.service.brandsafety;

import java.time.format.DateTimeFormatter;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.streaming.SXSSFSheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;

import ru.yandex.direct.excel.processing.model.brandsafety.BrandSafetyStatsExcelExportData;
import ru.yandex.direct.excel.processing.model.brandsafety.BrandSafetyStatsExcelExportLanguage;
import ru.yandex.direct.excel.processing.model.brandsafety.BrandSafetyStatsExcelExportRow;

import static ru.yandex.direct.excel.processing.model.brandsafety.BrandSafetyStatsExcelExportLanguage.EN;
import static ru.yandex.direct.excel.processing.model.brandsafety.BrandSafetyStatsExcelExportLanguage.RU;

@ParametersAreNonnullByDefault
public class BrandSafetyStatsExcelExportService {
    private static final Map<BrandSafetyStatsExcelExportLanguage, String> SHEET_NAME =
            Map.of(RU, "Мастер отчётов", EN, "Report wizard");
    private static final Map<BrandSafetyStatsExcelExportLanguage, String> DATE_TITLE = Map.of(RU, "Дата", EN, "Date");
    private static final Map<BrandSafetyStatsExcelExportLanguage, String> CAMPAIGN_NAME_TITLE = Map.of(
            RU, "Кампания", EN, "Сampaign");
    private static final Map<BrandSafetyStatsExcelExportLanguage, String> CAMPAIGN_ID_TITLE = Map.of(
            RU, "№ кампании", EN, "Сampaign №");
    private static final Map<BrandSafetyStatsExcelExportLanguage, String> CATEGORY_TITLE = Map.of(
            RU, "Категории", EN, "Categories");
    private static final Map<BrandSafetyStatsExcelExportLanguage, String> PERCENTAGE_TITLE = Map.of(
            RU, "Доля отфильтрованных показов, %", EN, "Filtered, %");
    private static final Map<BrandSafetyStatsExcelExportLanguage, String> DATE_STRING_PATTERN = Map.of(
            RU, "Всего с %s по %s", EN, "Total from %s to %s");
    private static final Map<BrandSafetyStatsExcelExportLanguage, String> HEADER_STRING_PATTERN = Map.of(
            RU, "Клиент %s (%s), период %s - %s", EN, "Client %s (%s), period %s - %s");
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("dd.MM.yyyy");
    private static final int WIDTH_MULTIPLIER = 256;

    public Workbook createWorkbook(BrandSafetyStatsExcelExportData exportData) {
        var workbook = new SXSSFWorkbook(1000_000);
        workbook.setCompressTempFiles(true);

        var columnsCount = 2 + (exportData.getShowCampaigns() ? 2 : 0) + (exportData.getShowCategories() ? 1 : 0);
        var sheet = workbook.createSheet(SHEET_NAME.get(exportData.getLanguage()));
        setColumnWidth(sheet, columnsCount, exportData);

        var rowNum = 0;
        rowNum = writeHeader(sheet, rowNum, exportData, getHeaderStyle(workbook));

        var statsStyle = getStatsStyle(workbook);
        rowNum = writeTotal(sheet, rowNum, columnsCount, exportData, statsStyle);
        sheet.createRow(rowNum++);
        rowNum = writeTitle(sheet, rowNum, exportData, getTitleStyle(workbook));

        var textStyle = getTextStyle(workbook);
        for (var statsRow : exportData.getRows()) {
            var excelRow = sheet.createRow(rowNum++);
            writeStatsRow(exportData, statsRow, excelRow, statsStyle, textStyle);
        }

        return workbook;
    }

    private void setColumnWidth(SXSSFSheet sheet, int columnsCount, BrandSafetyStatsExcelExportData exportData) {
        sheet.setColumnWidth(0, 12 * WIDTH_MULTIPLIER);

        if (exportData.getShowCampaigns()) {
            sheet.setColumnWidth(1, 12 * WIDTH_MULTIPLIER);
            sheet.setColumnWidth(2, 55 * WIDTH_MULTIPLIER);
        }

        if (exportData.getShowCategories()) {
            sheet.setColumnWidth(columnsCount - 2, 55 * WIDTH_MULTIPLIER);
        }

        sheet.setColumnWidth(columnsCount - 1, 23 * WIDTH_MULTIPLIER);
    }

    private static CellStyle getHeaderStyle(Workbook workbook) {
        var font = workbook.createFont();
        font.setFontHeightInPoints((short) 10);
        font.setBold(true);

        var style = workbook.createCellStyle();
        style.setFont(font);
        style.setAlignment(HorizontalAlignment.LEFT);

        return style;
    }

    private static CellStyle getStatsStyle(Workbook workbook) {
        var font = workbook.createFont();
        font.setFontHeightInPoints((short) 8);
        font.setBold(true);

        var style = workbook.createCellStyle();
        style.setFont(font);
        style.setAlignment(HorizontalAlignment.LEFT);

        return style;
    }

    private static CellStyle getTextStyle(Workbook workbook) {
        var font = workbook.createFont();
        font.setFontHeightInPoints((short) 10);

        var style = workbook.createCellStyle();
        style.setFont(font);
        style.setAlignment(HorizontalAlignment.LEFT);

        return style;
    }

    private static CellStyle getTitleStyle(Workbook workbook) {
        var style = getStatsStyle(workbook);
        style.setBorderBottom(BorderStyle.MEDIUM);
        return style;
    }

    private void writeStatsRow(
            BrandSafetyStatsExcelExportData exportData,
            BrandSafetyStatsExcelExportRow statsRow,
            Row row,
            CellStyle statsStyle,
            CellStyle textStyle
    ) {
        var columnNum = 0;
        addCell(textStyle, statsRow.getDateString(), row, columnNum++);

        if (exportData.getShowCampaigns()) {
            addCell(textStyle, statsRow.getCid().toString(), row, columnNum++);
            addCell(textStyle, statsRow.getCampaignName(), row, columnNum++);
        }

        if (exportData.getShowCategories()) {
            addCell(textStyle, statsRow.getCategoryName(), row, columnNum++);
        }

        addCell(statsStyle, statsRow.getPercentage().toString(), row, columnNum);
    }

    private int writeTitle(Sheet sheet, int rowNum, BrandSafetyStatsExcelExportData exportData, CellStyle style) {
        var row = sheet.createRow(rowNum);
        var columnNum = 0;
        addCell(style, DATE_TITLE.get(exportData.getLanguage()), row, columnNum++);

        if (exportData.getShowCampaigns()) {
            addCell(style, CAMPAIGN_ID_TITLE.get(exportData.getLanguage()), row, columnNum++);
            addCell(style, CAMPAIGN_NAME_TITLE.get(exportData.getLanguage()), row, columnNum++);
        }

        if (exportData.getShowCategories()) {
            addCell(style, CATEGORY_TITLE.get(exportData.getLanguage()), row, columnNum++);
        }

        addCell(style, PERCENTAGE_TITLE.get(exportData.getLanguage()), row, columnNum);
        return rowNum + 1;
    }

    private int writeTotal(
            Sheet sheet, int rowNum, int columnsCount, BrandSafetyStatsExcelExportData exportData, CellStyle style) {
        var dateRangeStr = String.format(
                DATE_STRING_PATTERN.get(exportData.getLanguage()),
                DATE_TIME_FORMATTER.format(exportData.getStartDate()),
                DATE_TIME_FORMATTER.format(exportData.getEndDate()));

        var row = sheet.createRow(rowNum);
        addCell(style, dateRangeStr, row, 0);
        addCell(style, exportData.getTotal().toString(), row, columnsCount - 1);
        return rowNum + 1;
    }

    private int writeHeader(Sheet sheet, int rowNum, BrandSafetyStatsExcelExportData exportData, CellStyle style) {
        var headerString = String.format(
                HEADER_STRING_PATTERN.get(exportData.getLanguage()),
                exportData.getUserName(),
                exportData.getUserLogin(),
                DATE_TIME_FORMATTER.format(exportData.getStartDate()),
                DATE_TIME_FORMATTER.format(exportData.getEndDate()));

        var row = sheet.createRow(rowNum);
        addCell(style, headerString, row, 0);
        return rowNum + 1;
    }

    private void addCell(CellStyle style, String value, Row row, int columnNum) {
        var cell = row.createCell(columnNum);
        cell.setCellValue(value);
        cell.setCellStyle(style);
    }
}
