package ru.yandex.direct.excel.processing.service.internalad;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DataValidation;
import org.apache.poi.ss.usermodel.DataValidationConstraint;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.ss.util.CellReference;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.service.TemplateInfoService;
import ru.yandex.direct.excel.processing.model.ColumnsWithChoices;
import ru.yandex.direct.excel.processing.model.internalad.mappers.AdGroupMapperSettings;
import ru.yandex.direct.excel.processing.model.internalad.mappers.BannerMapperSettings;

import static com.google.common.base.Preconditions.checkState;

/**
 * Сервис добавляет валидацию данных (aka drop-down list) для столбцов у которых есть ограничивающие значения
 * про drop-down list можно почитать/посмотреть тут:
 * https://support.office.com/en-us/article/create-a-drop-down-list-7693307a-59ef-400a-b769-c5402dce407b
 * <p>
 * Преамбула: столбцы сгруппированны по одинаковости ограничивающих значений, идентификатором столбца является заголовок
 * Алгоритм добавления валидации:
 * - Если у продукта нет фичи, то валидация не добавляется :)
 * - Для каждого листа с данными создается новый лист куда записываются данные для валидации
 * - Лист для валидации делается скрытым, чтобы не отвлекать пользователей: {@link #createValidationSheetFor}
 * - В этом листе на каждой строке записывается список ограничивающих значений:
 * {@link ValidationConstraintCreator#writeColumnsWithChoices}
 * - После записи каждой строки создается constraint, который указывает на эту строку:
 * {@link ValidationConstraintCreator#createConstraint}
 * - Далее на листе с данными создается диапазон для каждого столбца у которых ограниченные значения были записаны на
 * предыдущем шаге {@link CellRangeByTitleCreator#createFor}
 * - Делаяется привязка constraint'a с созданным диапазоном: {@link #linkConstraintToCellRange}
 */
@Service
@ParametersAreNonnullByDefault
public class AddExcelValidationDataService {

    private final TemplateInfoService templateInfoService;
    private final CryptaSegmentDictionariesService cryptaSegmentDictionariesService;

    @Autowired
    public AddExcelValidationDataService(TemplateInfoService templateInfoService,
                                         CryptaSegmentDictionariesService cryptaSegmentDictionariesService) {
        this.templateInfoService = templateInfoService;
        this.cryptaSegmentDictionariesService = cryptaSegmentDictionariesService;
    }

    public void addAdGroupsValidationData(Sheet sheet, int firstRow, List<String> columnTitles) {
        var columnsWithChoices = AdGroupMapperSettings.getColumnsWithChoices(cryptaSegmentDictionariesService);
        addValidationData(sheet, firstRow, columnTitles, columnsWithChoices);
    }

    public void addBannersValidationData(Sheet sheet, int firstRow, List<String> columnTitles, long templateId,
                                         boolean isModerated) {
        InternalTemplateInfo templateInfo = templateInfoService.getByTemplateId(templateId);
        List<ResourceInfo> resources = Objects.requireNonNull(templateInfo).getResources();

        var columnsWithChoicesList = BannerMapperSettings.getColumnsWithChoices(resources, isModerated);
        addValidationData(sheet, firstRow, columnTitles, columnsWithChoicesList);
    }


    /**
     * Добавляет валидацию данных для эксель
     *
     * @param sheet                  лист с данными для которых нужно добавить валидацию
     * @param firstRow               номер строки, с которой начинаются данные
     * @param columnTitles           заголовки столбцов. По заголовке понимаем для какого столбца нужно добавить
     *                               валидацию
     * @param columnsWithChoicesList список столбцов с ограниченными  значениями
     */
    private static void addValidationData(Sheet sheet, int firstRow, List<String> columnTitles,
                                          List<ColumnsWithChoices> columnsWithChoicesList) {
        var cellRangeCreator = new CellRangeByTitleCreator(columnTitles, firstRow, sheet.getLastRowNum());
        Sheet validationSheet = createValidationSheetFor(sheet);
        var constraintCreator = new ValidationConstraintCreator(validationSheet);

        for (ColumnsWithChoices columnsWithChoices : columnsWithChoicesList) {
            DataValidationConstraint constraint = constraintCreator.createFor(columnsWithChoices);

            for (String title : columnsWithChoices.getColumnTitles()) {
                CellRangeAddressList cellRange = cellRangeCreator.createFor(title);
                linkConstraintToCellRange(sheet, constraint, cellRange);
            }
        }
    }

    private static Sheet createValidationSheetFor(Sheet sheet) {
        Workbook workbook = sheet.getWorkbook();
        // VD - сокращенно от ValidationData. Имя листа ограниченно 31 символами, поэтому экономим)
        Sheet validationSheet = workbook.createSheet("VD_" + sanitizeSheetName(sheet.getSheetName()));
        workbook.setSheetHidden(workbook.getSheetIndex(validationSheet), true);
        return validationSheet;
    }

    /**
     * Заменяет все символы кроме букв и цифр на _
     * Нужно чтобы при использовании имени листа в формуле не было проблем.
     */
    private static String sanitizeSheetName(String sheetName) {
        return sheetName
                .replaceAll("[^\\w\\dА-Яа-я]", "_")
                .replaceAll("_+", "_");
    }

    private static void linkConstraintToCellRange(Sheet sheet, DataValidationConstraint constraint,
                                                  CellRangeAddressList cellRange) {
        var validationHelper = sheet.getDataValidationHelper();

        DataValidation dataValidation = validationHelper.createValidation(constraint, cellRange);
        dataValidation.setSuppressDropDownArrow(true);
        dataValidation.setShowErrorBox(true);

        sheet.addValidationData(dataValidation);
    }

    private static class ValidationConstraintCreator {

        private final Sheet validationSheet;

        private ValidationConstraintCreator(Sheet validationSheet) {
            this.validationSheet = validationSheet;
        }

        public DataValidationConstraint createFor(ColumnsWithChoices columnsWithChoices) {
            /*
            Определяем следующую строку, в которую можно писать.
            метод getLastRowNum() и для пустого листа, и для записанной первой строки -- выдаёт 0,
            поэтому отдельно проверяем не пуста ли первая строка
             */
            var validationRowIndex = validationSheet.getRow(0) == null
                    ? 0 : validationSheet.getLastRowNum() + 1;

            int columnNum = writeColumnsWithChoices(columnsWithChoices, validationRowIndex, validationSheet);

            return createConstraint(validationRowIndex + 1, columnNum);
        }

        private DataValidationConstraint createConstraint(int rowNum, int columnNum) {
            String columnAsString = CellReference.convertNumToColString(columnNum - 1);

            return validationSheet.getDataValidationHelper().createFormulaListConstraint(
                    String.format("%s!$B$%d:$%s$%d", validationSheet.getSheetName(), rowNum, columnAsString, rowNum));
        }

        private static int writeColumnsWithChoices(ColumnsWithChoices columnsWithChoices, int rowNum, Sheet sheet) {
            Row row = sheet.createRow(rowNum);

            int columnNum = 0;
            Cell titleCell = row.createCell(columnNum++, CellType.STRING);
            titleCell.setCellValue(titlesToString(columnsWithChoices.getColumnTitles()));

            for (String choice : columnsWithChoices.getChoices()) {
                Cell cell = row.createCell(columnNum++, CellType.STRING);
                cell.setCellValue(choice);
            }

            return columnNum;
        }

        private static String titlesToString(Set<String> titles) {
            return StreamEx.of(titles)
                    // сортируем для тестов
                    .sorted()
                    .joining(", ");
        }
    }

    private static class CellRangeByTitleCreator {

        private final int firstRow;
        private final int lastRowNum;
        private final Map<String, Integer> columnNumberByTitle;

        private CellRangeByTitleCreator(List<String> columnTitles, int firstRow, int lastRowNum) {
            this.firstRow = firstRow;
            this.lastRowNum = lastRowNum;
            this.columnNumberByTitle = EntryStream.of(columnTitles)
                    .invert()
                    .toImmutableMap();
        }

        public CellRangeAddressList createFor(String columnTitle) {
            checkState(columnNumberByTitle.containsKey(columnTitle), "unexpected title %s", columnTitle);
            Integer columnNumber = columnNumberByTitle.get(columnTitle);
            return new CellRangeAddressList(firstRow, lastRowNum, columnNumber, columnNumber);
        }
    }

}
