package ru.yandex.direct.excel.processing.service.internalad;

import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupAddItem;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupAddOrUpdateItem;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupUpdateItem;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.internal.ComplexInternalAdGroupService;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.ConvertResultOperation;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.aggregator.SplitAndMergeOperationAggregator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.result.PathHelper.field;

@Service
public class AddOrUpdateInternalAdGroupsService {

    private final ComplexInternalAdGroupService complexInternalAdGroupService;

    @Autowired
    public AddOrUpdateInternalAdGroupsService(ComplexInternalAdGroupService complexInternalAdGroupService) {
        this.complexInternalAdGroupService = complexInternalAdGroupService;
    }

    /**
     * комплексная операция добавления и обновления групп внутренней рекламы
     *
     * @param items              список групп для добавления и обновления. Если group.id != null, то данная группа будет
     *                           обновлена.
     * @param operationContainer Контейнер с параметрами для операций на разных уровнях вложенности
     * @param validationOnly     если true, то изменений в базе не будет
     * @return возвращает результат операции.
     */
    MassResult<Long> addOrUpdateInternalAdGroups(List<InternalAdGroupAddOrUpdateItem> items,
                                                 InternalAdGroupOperationContainer operationContainer,
                                                 boolean validationOnly) {

        var updateOperationCreator = getUpdateOperationCreator(operationContainer);
        var addOperationCreator = getAddOperationCreator(operationContainer);

        SplitAndMergeOperationAggregator<InternalAdGroupAddOrUpdateItem, Long> aggregatedAddOperation =
                SplitAndMergeOperationAggregator.builder()
                        .addSubOperation(getAdGroupToUpdatePredicate(), updateOperationCreator)
                        .addSubOperation(not(getAdGroupToUpdatePredicate()), addOperationCreator)
                        .build();

        if (validationOnly) {
            return aggregatedAddOperation.validate(items);
        }
        return aggregatedAddOperation.execute(items);
    }

    private Predicate<InternalAdGroupAddOrUpdateItem> getAdGroupToUpdatePredicate() {
        return item -> item.getAdGroup().getId() != null;
    }

    private OperationCreator<InternalAdGroupAddOrUpdateItem, Operation<Long>> getAddOperationCreator(
            InternalAdGroupOperationContainer operationContainer) {
        return inputItems ->
                new ConvertResultOperation<>(
                        complexInternalAdGroupService.getAddOperation(operationContainer, (mapList(inputItems,
                                AddOrUpdateInternalAdGroupsService::toInternalAdGroupAddItem))),
                        AddOrUpdateInternalAdGroupsService::convertMassResultForAdd
                );

    }

    private OperationCreator<InternalAdGroupAddOrUpdateItem, Operation<Long>> getUpdateOperationCreator(
            InternalAdGroupOperationContainer operationContainer) {
        return inputItems ->
                new ConvertResultOperation<>(
                        complexInternalAdGroupService.getUpdateOperation(operationContainer, mapList(inputItems,
                                AddOrUpdateInternalAdGroupsService::toInternalAdGroupUpdateItem)),
                        AddOrUpdateInternalAdGroupsService::convertMassResultForUpdate
                );
    }

    private static MassResult<Long> convertMassResultForAdd(MassResult<Long> result) {
        return new MassResult<>(
                mapList(result.getResult(), AddOrUpdateInternalAdGroupsService::convertResultForAdd),
                result.getValidationResult().transformUnchecked(new AddValidationTransformer()),
                result.getState());
    }

    private static MassResult<Long> convertMassResultForUpdate(MassResult<Long> result) {
        return new MassResult<>(
                mapList(result.getResult(), AddOrUpdateInternalAdGroupsService::convertResultForUpdate),
                result.getValidationResult().transformUnchecked(new UpdateValidationTransformer()),
                result.getState());
    }

    private static Result<Long> convertResultForAdd(Result<Long> result) {
        return new Result<>(result.getResult(), result.getValidationResult().transform(new AddTransformer()),
                result.getState());
    }

    private static Result<Long> convertResultForUpdate(Result<Long> result) {
        return new Result<>(result.getResult(), result.getValidationResult().transform(new UpdateTransformer()),
                result.getState());
    }

    private static InternalAdGroupAddItem toInternalAdGroupAddItem(InternalAdGroupAddOrUpdateItem addItem) {
        List<RetargetingConditionBase> retargetingConditions =
                addItem.getRetargetingCondition() != null ? List.of(addItem.getRetargetingCondition()) : emptyList();
        return new InternalAdGroupAddItem()
                .withAdGroup(addItem.getAdGroup())
                .withAdditionalTargetings(addItem.getAdditionalTargetings())
                .withRetargetingConditions(retargetingConditions);
    }

    private static InternalAdGroupUpdateItem toInternalAdGroupUpdateItem(InternalAdGroupAddOrUpdateItem updateItem) {
        List<RetargetingConditionBase> retargetingConditions =
                updateItem.getRetargetingCondition() != null ? List.of(updateItem.getRetargetingCondition()) :
                        emptyList();
        return new InternalAdGroupUpdateItem()
                .withAdGroupChanges(toModelChanges(updateItem.getAdGroup()))
                .withAdditionalTargetings(updateItem.getAdditionalTargetings())
                .withRetargetingConditions(retargetingConditions);
    }

    private static ModelChanges<InternalAdGroup> toModelChanges(InternalAdGroup item) {
        return new ModelChanges<>(item.getId(), InternalAdGroup.class)
                .process(item.getName(), InternalAdGroup.NAME)
                .processNotNull(item.getLevel(), InternalAdGroup.LEVEL)
                .process(item.getRf(), InternalAdGroup.RF)
                .process(item.getRfReset(), InternalAdGroup.RF_RESET)
                .process(item.getMaxClicksCount(), InternalAdGroup.MAX_CLICKS_COUNT)
                .process(item.getMaxClicksPeriod(), InternalAdGroup.MAX_CLICKS_PERIOD)
                .process(item.getMaxStopsCount(), InternalAdGroup.MAX_STOPS_COUNT)
                .process(item.getMaxStopsPeriod(), InternalAdGroup.MAX_STOPS_PERIOD)
                .process(item.getStartTime(), InternalAdGroup.START_TIME)
                .process(item.getFinishTime(), InternalAdGroup.FINISH_TIME)
                .process(item.getGeo(), InternalAdGroup.GEO);
    }

    private static InternalAdGroupAddOrUpdateItem toInternalAdGroupAddOrUpdateItem(InternalAdGroupAddItem addItem) {
        return new InternalAdGroupAddOrUpdateItem()
                .withAdditionalTargetings(addItem.getAdditionalTargetings())
                .withAdGroup(addItem.getAdGroup());
    }

    private static InternalAdGroupAddOrUpdateItem toInternalAdGroupAddOrUpdateItem(InternalAdGroupUpdateItem updateItem) {
        return new InternalAdGroupAddOrUpdateItem()
                .withAdditionalTargetings(updateItem.getAdditionalTargetings())
                .withAdGroup(updateItem.getAdGroupChanges().toModel());
    }

    @ParametersAreNonnullByDefault
    private static class UpdateValidationTransformer implements ValidationResult.ValidationResultTransformer<Defect> {
        @SuppressWarnings("unchecked")
        @Override
        public <OV> List<InternalAdGroupAddOrUpdateItem> transformValue(Path path, @Nullable OV oldValue) {
            List<InternalAdGroupUpdateItem> old = (List<InternalAdGroupUpdateItem>) oldValue;
            return mapList(old, o -> new UpdateTransformer().transformValue(path, o));
        }

        @Override
        public Map<PathNode, ValidationResult<?, Defect>> transformSubResults(
                Path path, Map<PathNode, ValidationResult<?, Defect>> subResults) {
            return EntryStream.of(subResults)
                    .<ValidationResult<?, Defect>>mapValues(subResult ->
                            subResult.transform(new UpdateTransformer()))
                    .toMap();
        }
    }

    @ParametersAreNonnullByDefault
    private static class UpdateTransformer implements ValidationResult.ValidationResultTransformer<Defect> {

        @Override
        public <OV> InternalAdGroupAddOrUpdateItem transformValue(Path path, @Nullable OV oldValue) {
            InternalAdGroupUpdateItem old = (InternalAdGroupUpdateItem) oldValue;
            return AddOrUpdateInternalAdGroupsService.toInternalAdGroupAddOrUpdateItem(old);
        }

        @Override
        public Map<PathNode, ValidationResult<?, Defect>> transformSubResults(
                Path path, Map<PathNode, ValidationResult<?, Defect>> subResults) {
            PathNode adGroupChangesPathNode = field(InternalAdGroupUpdateItem.AD_GROUP_CHANGES);
            PathNode adGroupPathNode = field(InternalAdGroupAddOrUpdateItem.AD_GROUP);
            if (subResults.containsKey(adGroupChangesPathNode)) {
                subResults.put(adGroupPathNode, subResults.get(adGroupChangesPathNode));
                subResults.remove(adGroupChangesPathNode);
            }
            return subResults;
        }
    }

    @ParametersAreNonnullByDefault
    private static class AddValidationTransformer implements ValidationResult.ValidationResultTransformer<Defect> {
        @SuppressWarnings("unchecked")
        @Override
        public <OV> List<InternalAdGroupAddOrUpdateItem> transformValue(Path path, @Nullable OV oldValue) {
            List<InternalAdGroupAddItem> old = (List<InternalAdGroupAddItem>) oldValue;
            return mapList(old, o -> new AddTransformer().transformValue(path, o));
        }

        @Override
        public Map<PathNode, ValidationResult<?, Defect>> transformSubResults(
                Path path, Map<PathNode, ValidationResult<?, Defect>> subResults) {
            return subResults;
        }
    }

    @ParametersAreNonnullByDefault
    private static class AddTransformer implements ValidationResult.ValidationResultTransformer<Defect> {

        @Override
        public <OV> InternalAdGroupAddOrUpdateItem transformValue(Path path, @Nullable OV oldValue) {
            InternalAdGroupAddItem old = (InternalAdGroupAddItem) oldValue;
            return AddOrUpdateInternalAdGroupsService.toInternalAdGroupAddOrUpdateItem(old);
        }

        @Override
        public Map<PathNode, ValidationResult<?, Defect>> transformSubResults(
                Path path, Map<PathNode, ValidationResult<?, Defect>> subResults) {
            return subResults;
        }
    }
}
